/* Copyright (C) 2002-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef __VNCVIEWERSDK_H__
#define __VNCVIEWERSDK_H__

#include "vnctypes.h"
#include "vnccallbacks.h"
#include "vncparameters.h"
#include "vncbearer.h"
#include "vncdecoder.h"
#include "vncrenderer.h"

/**
 * \file vncviewersdk.h
 *
 * This is the main header file for the VNC Viewer SDK.  To use the SDK,
 * you need only include this file.
 *
 * \see VNCViewerSDKInitialize, VNCViewerSDK
 */

/**
 * \mainpage VNC Viewer SDK - Native Interface
 *
 * \section section_contents Contents
 *
 *  -# \ref section_overview
 *  -# \ref section_usage
 *  -# \ref section_thread
 *  -# \ref section_keyevents
 *  -# \ref section_pointerevents
 *  -# \ref section_mirrorlink
 *  -# \ref section_skins
 *  -# \ref section_bearers
 *  -# \ref section_extensions
 *  -# \ref section_decoders
 *  -# \ref section_renderers
 *  -# \ref section_legal
 *
 * \section section_overview Overview
 *
 * The VNC Viewer SDK allows to you to create and manipulate VNC Viewer
 * objects.  A VNC Viewer can connect to a VNC Server and maintain a
 * 'framebuffer' whose contents reflect the appearance of the VNC Server host's
 * own display, allowing your application to render it.  The SDK also provides
 * APIs to transmit keyboard and pointer (i.e. mouse or touchscreen) events to
 * the VNC server.
 *
 * The native interface to the SDK does not provide any user interface of its
 * own.
 *
 * \section section_usage Basic usage
 *
 * \subsection subsection_usage_compiling Compiling your application
 *
 * To use the SDK, your application must first define exactly one of the
 * preprocessor macros VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS.  These
 * macros govern the definitions of the vnc_*_t integer types defined by
 * vncint.h.
 *
 *  - If your build environment provides the ISO C99 header stdint.h, or you
 *    can author an equivalent yourself, you should define VNC_USE_STDINT_H.
 *    This is the recommended approach for defining the SDK's integer types.
 *  - If you examine vncint.h and find that the types used when you define
 *    VNC_USE_DEFAULT_INT_TYPEDEFS are correct for your platform, then you may
 *    wish to define VNC_USE_DEFAULT_INT_TYPEDEFS instead of VNC_USE_STDINT_H.
 *    Additionally, if you are targeting a 64-bit platform with a compiler
 *    other than Microsoft Visual C++ or gcc, then you may have to define
 *    VNC_FORCE_64_BIT in order to define vnc_intptr_t and vnc_uintptr_t
 *    correctly.  See vncint.h itself for further information.
 *
 * The types defined by VNC_USE_DEFAULT_INT_TYPEDEFS are known to be correct
 * for Microsoft Visual C++ 2005.
 *
 * After defining either VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS, your
 * application should include vncviewersdk.h.  You are encouraged to use the
 * VNCVerifyIntegerTypes() macro to verify that the integer types have been
 * defined correctly.
 *
 * \subsection subsection_usage_startup Initializing the SDK
 *
 * -# Load the SDK DLL (necessary only if you did not link against it).
 *    - on Windows, call LoadLibrary()
 *    - on Linux, call dlopen()
 * -# Locate the SDK entry point, VNCViewerSDKInitialize().
 *    - on Windows, call GetProcAddress()
 *    - on Linux, call dlsym()
 * -# Call VNCViewerSDKInitialize().  This populates a VNCViewerSDK structure
 *    with pointers to the rest of the SDK's 'methods'.
 *
 * Typically, your application will call VNCViewerSDKInitialize() at start-up.
 * You should not call VNCViewerSDKInitialize() more than once.
 *
 * \subsection subsection_usage_creating Creating a VNC Viewer object
 *
 * Call VNCViewerCreate() to create a VNC Viewer object. You may reuse the same
 * VNC Viewer object for a succession of VNC sessions that require a similar
 * configuration. There is no need to create a new VNC Viewer for each session.
 *
 * If you need to run more than one VNC session in parallel, you can call
 * VNCViewerCreate() as often as needed so that you have a different VNC Viewer
 * instance for each session.  However, note that you should not call
 * VNCViewerSDKInitialize() more than once.
 *
 * When you create a VNC Viewer object, you supply pointers to various
 * callbacks that provide your application with information about VNC events.
 * You should always implement at least these two callbacks:
 *
 * - VNCViewerSessionProgressCallback(), so that you are notified when a
 *   connection is successfully established.
 * - VNCViewerErrorCallback(), so that you are notified when a connection
 *   attempt is unsuccessful or a connection is lost.
 *
 * If you are connecting outwards to the server and you think that the
 * server might require authentication, you should implement:
 *
 * - VNCViewerAsynchronousCredentialsCallback(), so that the user can provide a
 *   username and password if the server requires them.
 *
 * If your application is to render the framebuffer locally, you should
 * implement:
 *
 * - VNCViewerServerInitCallback(), so that you are told the initial size of
 *   the required framebuffer.
 * - VNCViewerDesktopResizeCallback(), so that you are told if the required
 *   size for the framebuffer changes.
 * - VNCViewerLockRectangleExCallback() and VNCViewerUnlockRectangleExCallback(),
 *   so that the SDK can be given safe access to your application's
 *   framebuffer.
 * - VNCViewerFrameBufferUpdateEndExCallback(), so that you can arrange for the
 *   updated region of your application's framebuffer to be redrawn.
 *
 * After creating a VNC Viewer object, you may find it useful to use
 * VNCViewerSetContext() to associate some application-specific data with the
 * viewer. For example, you could pass a pointer to a struct that holds state
 * for your application (window handles, log file handle, etc). It is
 * recommended that any call to VNCViewerSetContext() is made immediately after
 * VNCViewerCreate(), so that the context pointer is available to all callbacks
 * that the SDK may make.
 *
 * \subsection subsection_usage_licensing Licensing the viewer
 *
 * In order to use the SDK, the application must always provide each VNC Viewer
 * instance with at least one license. Use VNCViewerAddLicense() to do this.
 * Contact your RealVNC sales representative to obtain a license.
 *
 * \subsection subsection_usage_configuring Configuring the viewer
 *
 * Use VNCViewerSetParameter() to set any desired parameters on the VNC Viewer,
 * possibly determining their values by displaying UI to the end user. See
 * vncparameters.h for a full list of configuration options.
 *
 * Use VNCViewerSetSecurityTypes() to configure the encryption and
 * authentication mechanisms used by the VNC Viewer. If you intend to support
 * RA2 or RA2ne security, you must set ::VNCParameterRSAKeys (see
 * VNCGenerateRSAKeys() for more information).
 *
 * \subsection subsection_usage_connecting Connecting to a VNC server
 *
 * The preferred method for a session with a VNC is to format a
 * \ref vnccommandstring.h "VNC Command String"
 * and then call VNCViewerProcessCommandString().  VNC Command Strings
 * encapsulate the information (e.g. bearer type, IP address and port) required
 * to establish a connection to a VNC Server, and allow this information to be
 * specified by some external agent, such as a web application.
 *
 * The convenience APIs VNCViewerConnect() and VNCViewerListen() may be used as
 * shortcuts for connecting outwards over TCP and listening for incoming TCP
 * connections.  Internally, each of these APIs makes a call to
 * VNCViewerProcessCommandString().
 *
 * Calling any of these methods starts a new thread of execution.  See
 * \ref section_thread for further details.
 *
 * For details of individual errors, see the documentation relating to
 * that error code in vnctypes.h.
 *
  \subsection subsection_usage_rendering Rendering the framebuffer
 *
 * To render the VNC Server's display locally, your application must create and
 * maintain a framebuffer.  This is an area of memory into which the SDK will
 * copy display updates that it received from the VNC Server, and which your
 * application can draw onto the screen when necessary.
 *
 * Many software platforms that provide a drawing library have a type that
 * represents an in-memory bitmapped image.  It is often convenient to use one
 * of these as the framebuffer.  For example, a Win32 application might use a
 * device-independent bitmap (see CreateDIBSection()), while a .NET application
 * might use an instance of System.Drawing.Bitmap.
 *
 * Your application informs the SDK of the format of your framebuffer when it
 * calls VNCViewerCreate().  The framebuffer pixel format is then fixed for the
 * lifetime of the VNC Viewer object.
 *
 * Whatever your choice of framebuffer, and whatever its format, it must obey
 * these rules:
 *
 * - Each pixel must be aligned on a boundary suitable for storing a word of
 *   the pixel size.  (The size of each pixel is given by the framebuffer pixel
 *   format's bitsPerPixel / 8.)
 * - The pixels forming each row must be contiguous in memory.
 * - The rows must be evenly spaced in memory.  (The difference in bytes
 *   between the addresses of the first pixels in adjacent rows is known as the
 *   framebuffer's stride.  This is often, but not always, equal to the width
 *   of the framebuffer multiplied by the pixel size.)
 * - The pixel at (0, 0) is at the top-left and is the first pixel in the
 *   framebuffer.
 *
 * You should create a framebuffer whenever the SDK calls your
 * VNCViewerServerInitCallback() or VNCViewerDesktopResizeCallback().  Your
 * VNCViewerDesktopResizeCallback() should also destroy the old framebuffer.
 *
 * Depending on the server platform, the available bandwidth, the sizes of the
 * viewer and server displays, and the CPU resources available to the viewer and
 * server applications, it may be advantageous to enable server-side-scaling
 * from within the VNCViewerServerInitCallback. See
 * VNCViewerSetServerSideScaling() for a discussion of this feature.
 *
 * When a display update is received from the VNC Server, the SDK calls your
 * VNCViewerLockRectangleExCallback() to gain exclusive access to a section of
 * the framebuffer.  The SDK then converts the data received from the VNC
 * Server into the format required by your application's framebuffer, and
 * copies the data into the framebuffer.  Finally, the SDK calls your
 * VNCViewerUnlockRectangleExCallback(), at which point you should cause the
 * framebuffer to be rendered to the local display (e.g. by 'invalidating' it).
 *
 * A VNC server begins by sending updates in the server's native pixel format.
 * This is provided to your application by the VNCViewerServerInitCallback().
 * However, you can select a new pixel format for the session by calling
 * VNCViewerSetPixelFormat().  For example, you may wish to use a pixel format
 * with fewer colors, in order to save bandwidth.  You can query the current
 * session pixel format with VNCViewerGetPixelFormat().
 *
 * \subsection subsection_usage_destroying Destroying a VNC Viewer object
 *
 * Use VNCViewerDestroy() to destroy the VNC Viewer.  If the viewer thread is
 * still running, then VNCViewerDestroy() will call VNCViewerReset() for you.
 * Note that you must never call VNCViewerDestroy() directly from an SDK
 * callback, since this will result in a deadlock.
 *
 * \subsection subsection_usage_shutdown Uninitializing and unloading the SDK
 *
 * -# Call VNCViewerSDKUninitialize().
 * -# Unload the SDK (necessary only if you did not link against it).
 *    - on Windows, call FreeLibrary()
 *    - on Linux, call dlclose()
 *
 * \section section_thread The viewer thread
 *
 * Calling VNCViewerConnect(),  VNCViewerListen() or
 * VNCViewerProcessCommandString() starts a new thread of execution.  This
 * thread is called the viewer thread.  (In the case of
 * VNCViewerProcessCommandString(), the viewer thread is only started if the
 * call is successful.)
 *
 * Once started, the viewer thread continues to execute until one of the
 * following occurs:
 *
 * - The connection attempt is unsuccessful
 * - The established connection is lost
 * - Your application calls VNCViewerReset() (including implicit calls to
 *   VNCViewerReset() made by VNCViewerDestroy())
 *
 * Before it exits, the viewer thread will always make exactly one call to
 * your application's \ref VNCViewerErrorCallback.  This call will take place
 * even if your application calls VNCViewerReset().
 *
 * \subsection subsection_thread_app Interaction with your application
 *
 * All SDK processing for each viewer instance takes place within the viewer
 * thread for that viewer.  This includes processing that your application
 * requests by calling viewer APIs.  The majority of viewer APIs place a
 * command on a queue that the viewer thread services, and do not return to the
 * caller until the command has been serviced by the viewer thread.
 *
 * Almost all callbacks that the SDK makes into your application are always
 * made by the viewer thread.  This means that you should take care to protect
 * against simultaneous access to any application state that is shared between
 * the SDK callbacks and your own thread, e.g. with a mutex.  The most common
 * use of a mutex within a Viewer SDK application is to protect the
 * framebuffer, which is updated in the viewer thread by the lock/unlock
 * callbacks, but which will almost always be rendered by an application
 * thread.
 *
 * The only exception to this rule is the VNCViewerLogCallback(), which the SDK
 * may make from any thread, including calls made before the viewer thread is
 * started. However, the SDK guarantees that no two threads will enter a
 * VNCViewerLogCallback() at the same time, even if there are multiple viewer
 * instances.
 *
 * This threading model greatly simplifies the interaction of the SDK with your
 * application.  It does, however, mean that there are three particular rules
 * that your application must follow in order to avoid deadlock.
 *
 * \subsection subsection_thread_rule1 Don't call SDK API while holding locks
 *
 * For example, suppose that an application protects its framebuffer with a
 * mutex, as above.  If an application thread locks the framebuffer to render
 * it, and then queries \ref VNCViewerPropertyServerDesktopWidth or
 * \ref VNCViewerPropertyServerDesktopHeight while it still owns the lock, then
 * the likelihood of deadlock is very high.
 *
 * Sooner or later, an attempt by the viewer thread to claim the framebuffer
 * mutex will block because the application owns the lock.  While it is
 * blocked, the viewer thread will be unable to service the application
 * thread's API calls, and the application is deadlocked.
 *
 * \subsection subsection_thread_rule2 Don't block the viewer thread to wait for a response from an application thread
 *
 * It is common to want to communicate events from the viewer thread to an
 * application thread, so that UI can be updated.  For example, your
 * application may wish to use the \ref VNCViewerFrameBufferUpdateEndExCallback
 * as its cue to render the framebuffer, but your UI framework may require that
 * the framebuffer is rendered from an application thread.
 *
 * If the inter-thread RPC mechanism used to communicate this event to the
 * application thread blocks until it is serviced, then the likelihood of
 * deadlock is very high.  Sooner or later, the application thread will be
 * blocked on an SDK API call at the time that the viewer thread needs it to
 * service the blocking RPC, and the application is deadlocked.
 *
 * Common RPC mechanisms that should <i>not</i> be used from viewer callbacks
 * are the Win32 API SendMessage and the .NET Windows Forms method
 * Control.Invoke.  You should use PostMessage or Control.BeginInvoke instead.
 *
 * \subsection subsection_thread_rule3 Don't attempt to destroy a VNC Viewer from an SDK callback
 *
 * VNCViewerDestroy() shall block until the viewer thread has terminated. If
 * you call this API from an SDK callback, then the viewer thread shall wait
 * for itself to terminate, resulting in deadlock.
 *
 * If you wish to destroy a VNC Viewer instance in response to an SDK callback,
 * then your callback implementation should signal your application thread to
 * invoke VNCViewerDestroy().
 *
 * \section section_keyevents Key events
 *
 * Your application should use one or more of VNCViewerSendKeyEvent(),
 * VNCViewerProcessWindowsKeyMessage(), VNCViewerSendDeviceKeyEvent()
 * VNCViewerSendDeviceKeyOrKeyEvent() and VNCViewerSendPointerEvent() to
 * provide keyboard input  events to the VNC Viewer. Usually, you will call
 * these in response to keyboard events generated by the user of the
 * application.
 *
 * VNCViewerSendKeyEvent() is the most basic API. It always sends a single
 * RFB KeyEvent message to the VNC Server.
 *
 * VNCViewerSendDeviceKeyEvent() sends keys whose names are represented in the
 * ::VNCDeviceKey enumeration. These are keys that are commonly found on mobile
 * devices but are not defined as standard X key symbols.
 *
 * VNCViewerSendDeviceKeyOrKeyEvent() is typically used only with VNC Mobile
 * Servers. It maps PC keyboard keys to device keys, so that a PC can be used
 * to control a mobile device. If the server is not a VNC Mobile Server, then
 * it behaves as VNCViewerSendKeyEvent().
 *
 * VNCViewerProcessWindowsKeyMessage() is used only under Win32 or Windows CE.
 * It handles the keyboard-related Windows messages (WM_KEYDOWN, WM_KEYUP,
 * WM_SYSKEYDOWN, WM_SYSKEYUP and WM_CHAR) and translates them into calls to
 * either VNCViewerSendDeviceKeyOrKeyEvent() or VNCViewerSendKeyEvent().
 *
 * \subsection subsection_keyevents_repeat Single presses vs auto-repeats
 *
 * To communicate a single press of a key (or a device key; see below), the
 * viewer application should first send a 'down' key event to the VNC server,
 * and then an 'up' key event for the same key. It is important that the 'up'
 * event is always sent, even if the viewer application loses the input focus
 * or the viewer application exits.
 *
 * To communicate an auto-repeated key event to the VNC server, the viewer
 * application should:
 *
 *  - send a 'down' key event for that key
 *  - continue to send 'down' key events for that key while the key is held
 *    down
 *  - send a single 'up' key event when the key is released
 *
 * The viewer application should <i>not</i> send 'up' key events while the key
 * is being held down. Sending an 'up' key event will cause the server to
 * behave as though the key has been released, and the next 'down' event will
 * cause the server to behave as though the key has been re-pressed.
 *
 * Most GUI environments either provide a compatible 'many downs, one up'
 * pattern of key events, or can be asked to provide it. For example:
 *
 *  - Win32 and WinCE: default behavior of WM_KEYDOWN and WM_KEYUP
 *  - Android: default behavior of View.KeyListener
 *  - X11: not default; call XkbSetDetectableAutoRepeat()
 *  - Qt 4: not available; call XkbSetDetectableAutoRepeat() if available and
 *    appropriate
 *
 * If a sequence of correctly auto-repeated key events is sent to either a VNC
 * Mobile Server or a MirrorLink server, then the server should interpret the
 * entire key event sequence as a 'press and hold' or 'long press' of that key.
 *
 * \subsubsection subsection_keyevents_ml MirrorLink
 *
 * MirrorLink sessions always use plain RFB KeyEvent messages. You may
 * therefore prefer to use use only VNCViewerSendKeyEvent() with MirrorLink
 * sessions.
 *
 * However, you can still use VNCViewerSendDeviceKeyEvent(),
 * VNCViewerProcessWindowsKeyMessage(), and VNCViewerSendDeviceKeyOrKeyEvent()
 * when connected to a MirrorLink server. These methods translate any generated
 * device key events into the corresponding MirrorLink X key symbol, and then
 * behave as per VNCViewerSendKeyEvent(). This approach is particularly useful
 * if your viewer application must interoperate with both MirrorLink and
 * RealVNC servers.
 *
 * \subsubsection subsection_keyevents_cad Ctrl-Alt-Del
 *
 * There are two ways to send the special key sequence Ctrl-Alt-Del to the
 * server:
 *
 *  -# Write code to do it explicitly:
 *     \code
 *     VNCViewerSendKeyEvent(pViewer, XK_Control_L, TRUE);
 *     VNCViewerSendKeyEvent(pViewer, XK_Alt_L, TRUE);
 *     VNCViewerSendKeyEvent(pViewer, XK_Delete, TRUE);
 *     VNCViewerSendKeyEvent(pViewer, XK_Delete, FALSE);
 *     VNCViewerSendKeyEvent(pViewer, XK_Alt_L, FALSE);
 *     VNCViewerSendKeyEvent(pViewer, XK_Control_L, FALSE);
 *     \endcode
 *     and then hook it to an application menu or hotkey.
 *  -# Document that the user should press Shift-Ctrl-Alt-Del when they want to
 *     to do this.  Windows will not interpret this as the Secure Attention
 *     Sequence, but will pass the keys to the application as it would
 *     normally.  When the keys are then sent to the VNC server, however, the
 *     server will simulate a server-side Secure Attention Sequence, as if the
 *     user had pressed only Ctrl-Alt-Del locally.
 *
 * \section section_pointerevents Pointer events
 *
 * VNCViewerSendPointerEvent() is used to send pointer events to the VNC
 * server.
 *
 * If the viewer machine has a mouse, then the application should call
 * VNCViewerSendPointerEvent() for every mouse event generated by the platform
 * while the mouse is over the viewer part of the UI, even if there are no
 * buttons pressed (represent this with VNCPointerDeviceButtonNone). If the
 * mouse pointer leaves the viewer part of the UI while a mouse button is held
 * down, then it is usually best to continue to send pointer events until the
 * last button is released. This gives intuitive behavior for drag or scroll
 * operations that are initiated by the user.
 *
 * If the viewer machine has a touch-screen, then the finger touching the
 * display is represented by VNCPointerDeviceButton1. VNCPointerDeviceButton1
 * pointer events should be sent to the VNC server while the display is
 * touched. When the display is no longer touched, then a
 * VNCPointerDeviceButtonNone pointer event must be sent to the server.
 *
 * TouchEvent MirrorLink extension messages can be sent to MirrorLink servers
 * using VNCViewerSendTouchEvent(). This API can also be used to inject multi-
 * touch events on a RealVNC RFB4 Android server, the other RealVNC RFB4
 * servers do not yet support multi-touch events. In order to prevent unexpected
 * behaviour avoid mixing calls to VNCViewerSendTouchEvent() with calls to
 * VNCViewerSendPointerEvent() in the same connection.
 *
 * \section section_mirrorlink MirrorLink support
 *
 * The Viewer SDK provides support for the VNC elements of MirrorLink&reg;.
 * For a full discussion of MirrorLink support in the Viewer SDK, please
 * see vncmirrorlink.h.
 *
 * \section section_skins Mobile device skins
 *
 * A skin is a photorealistic, interactive representation of a device that is
 * rendered by the Viewer SDK into the application's framebuffer.  The viewer
 * user may click on the skin to simulate presses of the device's own buttons.
 * The Viewer SDK allows an application to choose which skin to use with a VNC
 * session, either before or after the session is established.
 *
 * To select a skin, call VNCViewerSetSkin().  To the calling application,
 * skinned sessions appear almost identical to unskinned sessions.  The skin is
 * rendered into the application's framebuffer between calls to the usual
 * VNCViewerLockRectangleExCallback() and VNCViewerUnlockRectangleExCallback().
 * Once the session is established, the SDK checks whether injected pointer
 * events correspond to skin buttons.  If so, it re-renders the button with a
 * highlight, and, if the button is pressed or released, it generates the
 * appropriate key event.
 *
 * The only difference to the application is that the SDK will instruct the
 * application to make the framebuffer equal to the size of the active skinface
 * image.  If the application needs to know the size or location of the
 * unskinned framebuffer within the framebuffer, it may use the
 * ::VNCViewerPropertyServerDesktopX, ::VNCViewerPropertyServerDesktopY,
 * ::VNCViewerPropertyServerDesktopWidth and
 * ::VNCViewerPropertyServerDesktopHeight properties.
 *
 * The SDK also allows the application to select which skin to use for a
 * particular session based on information returned by the VNC Server during
 * the RFB handshake.  This allows for the application to choose a skin that
 * matches the server host device exactly (or, if an exact match is not
 * available, matches it closely).
 *
 * The application should base its decision on the values of the
 * ::VNCViewerPropertyServerHostManufacturer,
 * ::VNCViewerPropertyServerHostModel and ::VNCViewerPropertyServerHostVersion
 * properties.  The best way to do this is by using
 * VNCViewerAddSkinFetchCallback() to register one or more
 * VNCViewerSkinFetchCallback() implementations.  The SDK will invoke the
 * registered VNCViewerSkinFetchCallback()s as soon as the device has
 * identified itself.
 *
 * At present, the skins cannot be used at the same time as pluggable
 * renderers.
 *
 * \section section_bearers Pluggable bearers
 *
 * The Viewer SDK provides support for pluggable bearers.  Pluggable bearers
 * allow your application to add support for extra transport layers or
 * encapsulating protocols. Some platforms do not support dynamic loading of
 * libraries. In this case ::VNCViewerRegisterBearer() should be used to
 * statically register the bearer.
 *
 * \ref vnccommandstring.h "VNC Command Strings"
 * provide the bearer with the configuration necessary to connect to the VNC
 * Server.  In particular, the "t" field gives the name of the bearer that
 * should be used.  For example, "t=C" in the VNC Command String selects the
 * "C" bearer, which makes outgoing connections over TCP/IP.
 *
 * For details of how to implement a pluggable bearer, see vncbearer.h.
 *
 * For details of how the SDK loads bearers, see VNCParameterBearerDirectory.
 *
 * \section section_extensions Application-defined extension messages
 *
 * The Viewer SDK provides support for application-defined protocol extensions.
 * When connecting to a compatible VNC Server, your application may use these
 * extensions to exchange messages with arbitrary content.
 *
 * To register a protocol extension, call VNCViewerRegisterExtension().  After
 * connecting to a VNC Server, the VNCViewerExtensionEnabledCallback() will
 * notify the application if the server supports the extension.
 *
 * Once the server has enabled the extension, the application can use
 * VNCViewerSendExtensionMessage() to send a message on the extension, and will
 * be notified of any messages received on the extension via the
 * VNCViewerExtensionMessageReceivedCallback().
 *
 * Note that application-defined protocol extensions are not supported under
 * RFB 3 sessions (including MirrorLink sessions).
 *
 * \section section_decoders Pluggable decoders
 *
 * The Viewer SDK provides support for pluggable decoders.  Pluggable
 * decoders allow your application to add support for decoding
 * compressed pixel data via custom decoders.  The pluggable decoder
 * can make use of custom hardware that is available to your
 * application to improve the decoding performance.
 *
 * A pluggable decoder can be registered by calling
 * ::VNCViewerRegisterDecoder() or ::VNCViewerLoadDecoderFromFile(),
 * depending on if it is provided as an external library or not.
 *
 * For further details about implementing a pluggable decoder, see
 * vncdecoder.h
 *
 * \section section_renderers Pluggable renderers
 *
 * The Viewer SDK provides support for pluggable renderers. Pluggable
 * renderers allow your application to render arbitrary data from pluggable
 * decoders, and bypass the Viewer SDK's internal pixel format conversion and
 * framebuffer compositing. This can lead to performance improvements when
 * rendering to a hardware overlay, and makes it easy to combine rectangles
 * outputted by pluggable decoders with rectangles decoded internally within
 * the SDK.
 *
 * A pluggable renderer may choose to display an overlay, in which case your
 * application can choose to stop rendering the framebuffer. If your
 * application continues rendering the framebuffer, it is harmless but the
 * rendering will be invisible. When a pluggable renderer stops displaying
 * an overlay, your application must continue rendering the framebuffer.
 * The SDK will invoke the \ref VNCViewerFramebufferStateChangedCallback to
 * notify your application whether it is necessary to render the framebuffer.
 *
 * A pluggable renderer can be registered by calling
 * ::VNCViewerRegisterRenderer() or ::VNCViewerLoadRendererFromFile(),
 * depending on if it is provided as an external library or not.
 *
 * For further details about implementing a pluggable renderer, see
 * vncrenderer.h
 *
 * \section section_legal Legal information
 *
 * Copyright &copy; 2002-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * Details of and copyright notices for third-party software that is used by
 * the VNC Viewer SDK can be found in the file Acknowledgements.txt in the SDK
 * distribution.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * MirrorLink is a registered trademark of Car Connectivity Consortium in the
 * U.S. and in other countries.
 *
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCViewerSDK, VNCViewerCallbacks, VNCViewerSDKInitialize,
 * VNCViewerSDKUninitialize, VNCViewerCreate, VNCViewerDestroy,
 * VNCViewerConnect, VNCViewerListen, VNCViewerProcessCommandString,
 * VNCViewerReset, VNCVerifyIntegerTypes
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Creates a new VNC Viewer instance.
 *
 * This method creates and returns a new, uninitialized VNC Viewer instance.
 * All of the viewer's parameters have their default values, and the viewer
 * thread will not be started until you call VNCViewerConnect(),
 * VNCViewerListen() or VNCViewerProcessCommandString().
 *
 * Before creating a VNC Viewer object, you must decide on the format of the
 * framebuffer that you will use.  This is independent of the pixel format that
 * is used for the RFB session; the SDK handles any conversion between the two.
 * You should therefore choose a framebuffer pixel format that is easy for your
 * application to support.
 *
 * Although your application should create a new framebuffer whenever its
 * VNCViewerDesktopResizeCallback() is invoked, the format of the framebuffers
 * is fixed when VNCViewerCreate() is called, and cannot be changed for the
 * lifetime of the VNC Viewer object.
 *
 * You are free to call VNCViewerConnect(), VNCViewerListen() and
 * VNCViewerReset() multiple times on the same VNC Viewer instance.  For
 * example, you can give the user the opportunity to attempt to reconnect to a
 * server after entering bad credentials.
 *
 * The Viewer SDK does not impose an upper limit on the number of VNC Viewer
 * instances that may exist at the same time.  You are free to call
 * VNCViewerCreate() as often as you like.
 *
 * \param pCallbacks Pointer to a VNCViewerCallbacks structure holding pointers
 * to callbacks that may be invoked by the SDK.  The SDK copies the contents of
 * *pCallbacks, so it is safe to free this structure once VNCViewerCreate()
 * returns.  Passing pCallbacks equal to NULL is equivalent to passing
 * pCallbacks non-NULL with every pointer in *pCallbacks set to NULL.
 * \param callbacksSize The size of *pCallbacks.  If pCallbacks is non-NULL,
 * pass sizeof(VNCViewerCallbacks); if pCallbacks is NULL, pass 0.
 * \param pFrameBufferPixelFormat The format of your application's framebuffer.
 *
 * \return The new VNC Viewer.  When you are finished with it, call
 * VNCViewerDestroy() to destroy it.  If a VNC Viewer could not be created,
 * this method returns NULL.
 *
 * \see VNCViewerDestroy, VNCViewerCallbacks, VNCPixelFormat
 */
typedef VNCViewer *VNCCALL
VNCViewerCreate(VNCViewerCallbacks *pCallbacks,
                size_t callbacksSize,
                const VNCPixelFormat *pFrameBufferPixelFormat);

/**
 * \brief Destroy an instance of the VNC Viewer.
 *
 * This method destroys a VNC Viewer instance.  If the VNC Viewer is still
 * running, it will be stopped first.  After calling this method, you should
 * not attempt to interact further with the VNC Viewer.
 *
 * In an active MirrorLink connection, this method may block for a considerable
 * length of time as the viewer thread waits for the server to respond to the
 * MirrorLink ByeBye message. To avoid this blocking, it is recommended that
 * you instead call VNCViewerReset(). Once your application receives the
 * VNCViewerErrorCallback(), you can then signal your application thread to
 * call VNCViewerDestroy().
 *
 * Note that you must never call this method directly from an SDK callback,
 * since this will result in a deadlock.
 *
 * \param pViewer The VNC Viewer that should be destroyed.
 *
 * \see VNCViewerCreate
 */
typedef void VNCCALL
VNCViewerDestroy(VNCViewer *pViewer);

/**
 * \brief Begin running the viewer thread and attempt to connect to a VNC
 * server.
 *
 * This method creates and starts a new thread in which to run the VNC Viewer.
 * In that thread, the VNC Viewer will attempt to connect to the VNC server.
 *
 * If the connection attempt is successful, then
 * VNCViewerSessionProgressSessionEstablished will be notified via the
 * VNCViewerSessionProgressCallback().  Otherwise, the VNCViewerErrorCallback()
 * will be invoked to inform you of the reason for failure.
 *
 * \param pViewer The VNC Viewer that should make the connection attempt.
 * \param serverName The name of the VNC server to which the connection should
 * be made.  A VNC server name has one of the following formats:
 *  - hostname::port - where port is a TCP port number.
 *  - hostname:display - where display is a number between 0 and 99 inclusive.
 *    This is equivalent to hostname::(display + 5900).
 *  - hostname - equivalent to hostname:0 or hostname::5900.
 * You may use an IPv4 address instead of a hostname if you prefer.
 *
 * \note Prior to calling VNCViewerConnect(), you should call
 * VNCViewerSetParameter() as necessary to provide the VNC Viewer with its
 * initial configuration.
 * \note Calling this method when the viewer thread is already running has no
 * effect.
 *
 * \see VNCViewerErrorCallback, VNCViewerReset
 */
typedef void VNCCALL
VNCViewerConnect(VNCViewer *pViewer, const char *serverName);

/**
 * \brief Begin running the viewer thread and listen for incoming connections.
 *
 * This method creates and starts a new thread in which to run the VNC Viewer.
 * In that thread, the VNC Viewer will bind to the given TCP port number and
 * begin listening for an incoming connection from a VNC server.
 *
 * You may find it useful to have the viewer listen for incoming connections if
 * your servers are on machines that do not have permanent domain names or IP
 * addresses (e.g. they are mobile devices).
 *
 * If a connection from a VNC server is successfully accepted, then the
 * listening socket will be closed and
 * VNCViewerSessionProgressSessionEstablished will be notified via the
 * VNCViewerSessionProgressCallback().  Otherwise, the VNCViewerErrorCallback()
 * will be invoked to inform you of the reason for failure.
 *
 * After a connection is lost, the viewer thread is stopped, and the listening
 * port will not be reopened.  If you want to listen for another incoming
 * connection, you must call VNCViewerListen() again.
 *
 * \param pViewer The VNC Viewer that should be started.
 * \param pReserved Reserved for future use.  Should be NULL.
 * \param port The TCP port number on which to listen.  If this is 0, then the
 * default port number 5500 will be used.
 *
 * \note Prior to calling VNCViewerListen(), you should call
 * VNCViewerSetParameter() as necessary to provide the VNC Viewer with its
 * initial configuration.
 * \note Calling this method when the viewer thread is already running has no
 * effect.
 *
 * \see VNCViewerErrorCallback, VNCViewerReset
 */
typedef void VNCCALL
VNCViewerListen(VNCViewer *pViewer, void *pReserved, vnc_uint16_t port);

/**
 * \brief Parses the given VNC Command String and then executes the command.
 *
 * \ref vnccommandstring.h "VNC Command Strings"
 * allow the mechanism for establishing the session with the VNC Server to be
 * specified by some external agent, such as a web application.  In particular,
 * a VNC Command String is always required when connecting via a VNC Data
 * Relay.
 *
 * This method parses the given VNC Command String in the calling thread.  If
 * the string is parsed successfully, and the viewer thread is not running,
 * then the viewer thread will be started with instruction to process the new
 * command string. If the viewer thread is running, the method will fail with
 * VNCViewerErrorIllegalWhileRunning.
 *
 * If the string is not parsed successfully, or if it contains an invalid
 * combination of fields, then the method will fail with
 * VNCViewerErrorInvalidCommandString.
 *
 * For more information on VNC Command Strings, see vnccommandstring.h.
 *
 * If a VNC session is successfully established, then
 * VNCViewerSessionProgressSessionEstablished will be notified via the
 * VNCViewerSessionProgressCallback().  Otherwise, the VNCViewerErrorCallback()
 * will be invoked to inform you of the reason for failure.
 *
 * \param pViewer The VNC Viewer that should make the connection attempt.
 * \param commandString A VNC Command String.
 *
 * \retval VNCViewerErrorNone The command string was parsed successfully and the
 * viewer thread has been started.
 * \retval VNCViewerErrorIllegalWhileRunning The method has been invoked while
 * another instance of the viewer thread is running.
 * \retval VNCViewerErrorInvalidCommandString The command string is not valid.
 *
 * \note Prior to calling VNCViewerProcessCommandString(), you should call
 * VNCViewerSetParameter() as necessary to provide the VNC Viewer with its
 * initial configuration.
 *
 * \see VNCViewerErrorCallback, VNCViewerReset, vnccommandstring.h
 */
typedef VNCViewerError VNCCALL
VNCViewerProcessCommandString(VNCViewer *pViewer, const char *commandString);

/**
 * \brief Stop the viewer thread and, if connected to a VNC server, disconnect.
 *
 * If the viewer thread is running, this method will instruct the VNC Viewer to
 * close its connection to the server and then stop and destroy the viewer
 * thread.  Immediately before the viewer thread terminates, it will invoke
 * your VNCViewerErrorCallback() with VNCViewerErrorReset.
 *
 * Calling VNCViewerReset() does not destroy the VNC Viewer object itself.
 * You may either call VNCViewerDestroy() to destroy it, or, once it has
 * finished stopping, call VNCViewerConnect() or VNCViewerListen() to start it
 * again.
 *
 * Calling this method when the viewer thread is not running has no effect.
 *
 * \param pViewer The VNC Viewer that should be reset.
 *
 * \see VNCViewerDestroy
 */
typedef void VNCCALL
VNCViewerReset(VNCViewer *pViewer);

/**
 * \brief Queries a VNC Viewer for the value of an integer-valued runtime
 * property.
 *
 * You can use this method to find out information about a running VNC Viewer,
 * such as the dimensions of the framebuffer and a line speed estimate.
 *
 * To query string-valued runtime properties, use VNCViewerGetPropertyString()
 * instead.
 *
 * \param pViewer The VNC Viewer that should be queried.
 * \param property The property to query, such as framebuffer width or
 * height.
 *
 * \return Returns the value of the property.
 *
 * \see VNCViewerProperty
 */
typedef vnc_int32_t VNCCALL
VNCViewerGetProperty(VNCViewer *pViewer, VNCViewerProperty property);

/**
 * \brief Queries a VNC Viewer for the value of a string-valued runtime
 * property.
 *
 * You can use this method to find out information about a running VNC Viewer,
 * such as the type of mobile device to which it is connected.
 *
 * To query integer-valued runtime properties, use VNCViewerGetProperty()
 * instead.
 *
 * \param pViewer The VNC Viewer that should be queried.
 * \param property The property to query.
 *
 * \return Returns the value of the property, or NULL if the property value
 * could not be determined.  The caller is responsible for freeing the string
 * using VNCFreeString().
 *
 * \see VNCViewerProperty
 */
typedef char *VNCCALL
VNCViewerGetPropertyString(VNCViewer *pViewer,
                           VNCViewerProperty property);

/**
 * \brief Queries the value of a VNC Viewer parameter.
 *
 * The configuration of a VNC Viewer is stored internally as a list of
 * name-value pairs.  This method queries the value of a VNC Viewer parameter
 * and returns it.  The parameter names are case-insensitive.
 *
 * \param pViewer The VNC Viewer whose parameter to query.
 * \param parameter The name of the VNC Viewer parameter to set as a UTF-8
 * encoded, NUL-terminated string.  See vncparameters.h for an explanation of
 * the supported parameters.
 *
 * \return Returns a copy of the parameter's value as a UTF-8 encoded,
 * NUL-terminated string.  When you are finished with the copied value, you
 * must free it by calling VNCFreeString().  Returns NULL if the parameter name
 * is not valid or if memory could not be allocated.
 *
 * \note The values of parameters whose values are integers are formatted into
 * strings of decimal digits.
 *
 * \see VNCViewerSetParameter, vncparameters.h
 */
typedef char *VNCCALL
VNCViewerGetParameter(VNCViewer *pViewer, const char *parameter);

/**
 * Sets the value of a VNC Viewer parameter.
 *
 * If the viewer is running, then the new value will take effect immediately,
 * provided that this makes sense.  For example, a change to a connected
 * viewer's VNCParameterPreferredEncoding parameter will take effect
 * immediately.  However, it is meaningless to change a connected viewer's
 * VNCParameterRSAKeys parameter, because a session key has already been
 * negotiated.
 *
 * VNC Viewer parameters may never have NULL values.  To 'clear' the
 * value of a string parameter, pass an empty string as the value.
 *
 * Parameters whose values are integers are formatted in decimal.
 *
 * VNC Viewer parameter names are case-insensitive.
 *
 * \param pViewer The VNC Viewer whose parameter to set.
 * \param parameter The name of the VNC Viewer parameter to set as a UTF-8
 * encoded, NUL-terminated string.  See vncparameters.h for an explanation of
 * the supported parameters.
 * \param value The new parameter value as a UTF-8 encoded, NUL-terminated
 * string.  You may free this string after VNCViewerSetParameter() has
 * returned.
 *
 * \retval VNCViewerErrorNone The parameter was set successfully.
 * \retval VNCViewerErrorInvalidParameter Either the parameter name is not
 * recognized, or the given value is not valid for this parameter.
 *
 * \see VNCViewerGetParameter, vncparameters.h
 */
typedef VNCViewerError VNCCALL
VNCViewerSetParameter(VNCViewer *pViewer,
                      const char *parameter,
                      const char *value);

/**
 * \brief Sets a configuration for a bearer used by the VNC Viewer.
 *
 * Some bearers require a configuration to be set before a connection can be
 * established. This configuration should only contain settings that are static
 * for the lifetime of the viewer. Settings that vary between connections, such as
 * the address of a server to connect to, should not be included. The configuration
 * is specific to the VNC Viewer provided. Different VNC Viewer instances may set
 * different configurations for a particular bearer.
 *
 * The structure of the configuration is bearer-specific. A bearer is free to ignore
 * any configuration provided. If no configuration is provided, or the configuration
 * is invalid, then a bearer may refuse to establish a connection. No validation of
 * the configuration is performed by this method. This method may not be invoked
 * whilst a VNC session is active.
 *
 * \param pViewer The VNC Viewer that will be using the bearer.
 * \param bearerName The name of the bearer whose configuration is to be set.
 * \param configuration The configuration to provide to the bearer.
 * \retval VNCViewerErrorNone The configuration was set successfully.
 * \retval VNCViewerInvalidParameter The bearer name was NULL or empty string.
 * \retval VNCViewerErrorIllegalWhileRunning The operation failed because the
 * viewer thread is running.
 *
 * \see VNCBearerError, VNCBearerCreateConnection, VNCConnectionEstablish
 */
typedef VNCViewerError VNCCALL
VNCViewerSetBearerConfiguration(VNCViewer *pViewer,
                                const char *bearerName,
                                const char *configuration);

/**
 * \brief Configures the encryption and authentication mechanisms used by the
 * VNC Viewer.
 *
 * In the RFB handshake, a VNC Server advertises to the VNC Viewer the list of
 * security types that it is willing to use for the session.  The VNC Viewer
 * chooses a single security type, and then negotiation begins according to the
 * one that was chosen.  VNCViewerSetSecurityTypes() specifies the list of
 * security types, in order of preference, that the VNC Viewer is willing to
 * use.
 *
 * The default list of security types is (in decreasing order of preference):
 * RA2, RA2Rev, RA2ne, RA2Revne, VncAuth, None.  So, preference is given to an
 * encrypted session, followed by a session that encrypts credentials.
 * However, you are advised to have your application call
 * VNCViewerSetSecurityTypes() explicitly.  This is especially important if you
 * plan to call VNCViewerListen() instead of VNCViewerConnect(), because it
 * forces the VNC Server is comply with your own security requirements.
 *
 * VNCViewerSetSecurityTypes() must be called before the viewer thread is
 * started.
 *
 * \param pViewer The VNC Viewer whose security types to select.
 * \param securityTypes The array of security types to select (in decreasing
 * order of preference).
 * \param securityTypesCount The number of elements in the securityTypes array.
 * If securityTypesCount is zero, then the list of security types is reset to
 * the default.
 *
 * \retval VNCViewerErrorNone The list of security types was set successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The operation failed because the
 * viewer thread is running.
 * \retval VNCViewerErrorInvalidParameter One or more of the security types is
 * invalid or unsupported.
 *
 * \see VNCViewerConnect, VNCViewerListen
 */
typedef VNCViewerError VNCCALL
VNCViewerSetSecurityTypes(VNCViewer *pViewer,
                          const VNCSecurityType *securityTypes,
                          size_t securityTypesCount);

/**
 * \brief Iterates the names of the supported VNC Viewer parameters.
 *
 * \param pViewer The VNC Viewer whose parameter names to iterate.
 * \param pContext A structure that holds state between calls to
 * VNCViewerIterateParameterNames().  Before the first call to
 * VNCViewerIterateParameterNames(), allocate and zero the structure.  To
 * retrieve subsequent parameter names, call VNCViewerIterateParameterNames()
 * again without modifying the structure.
 * \param contextSize The size of the VNCParameterIterationContext structure.
 *
 * \return Returns the name of next parameter.  This string is owned by the
 * SDK; do not free it.  Return NULL if there are no more parameters to
 * iterate.
 *
 * \see VNCParameterIterationContext, VNCViewerGetParameter
 */
typedef const char *VNCCALL
VNCViewerIterateParameterNames(VNCViewer *pViewer,
                               VNCParameterIterationContext *pContext,
                               size_t contextSize);

/**
 * \brief Associates a context pointer with a VNC Viewer.
 *
 * Use this method to associate data with a VNC Viewer instance so that you can
 * recover it at a later time. The context pointer is supplied to all callbacks
 * made by the SDK, and can also be retrieved by calling VNCViewerGetContext().
 *
 * A second call to VNCViewerSetContext() replaces the context pointer assigned
 * by the first call.
 *
 * \param pViewer The viewer for which to set the context pointer.
 * \param pContext The context pointer to associate with the viewer.
 *
 * \note The SDK will not interact in any way with the memory to which the
 * context pointer points.  In particular, the SDK will not free the memory
 * when the VNC Viewer is destroyed.
 *
 * \see VNCViewerGetContext
 */
typedef void VNCCALL
VNCViewerSetContext(VNCViewer *pViewer, void *pContext);

/**
 * \brief Retrieves the VNC Viewer's context pointer.
 *
 * This method returns the context pointer associated with the VNC Viewer by a
 * call to VNCViewerSetContext().
 *
 * \param pViewer The viewer for which to retrieve the context pointer.
 * \return The viewer's context pointer, or NULL if there is none.
 *
 * \see VNCViewerGetContext
 */
typedef void *VNCCALL
VNCViewerGetContext(VNCViewer *pViewer);

/**
 * \brief Enables or disables display updates.
 *
 * This method can be used to prevent the VNC server from sending further
 * display updates to the given VNC Viewer object.  This is useful when it is
 * temporarily unnecessary to continue to update the VNC Viewer's display in
 * real time (for example, because it is obscured) because it prevents the VNC
 * Server from continuing to do unnecessary work.
 *
 * Re-enabling display updates will cause the VNC server to refresh the entire
 * display.  The entire display will be refreshed whenever
 * VNCViewerEnableDisplayUpdates(TRUE) is called, even if display updates were
 * never disabled.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param enable If non-zero, (re-)enable updates and refresh the entire
 * display; if zero, disable updates.
 *
 * \note Calls to the display update callbacks may continue for a few moments
 * after VNCViewerEnableDisplayUpdates(FALSE) is called.  This is because the
 * server was in the middle of an update.  Further updates will be disabled
 * once any outstanding updates have finished.
 *
 * \see VNCViewerPropertyDisplayUpdatesEnabled, VNCParameterRequestUpdatesEvent
 */
typedef void VNCCALL
VNCViewerEnableDisplayUpdates(VNCViewer *pViewer,
                              vnc_uint8_t enable);

/**
 * \brief Sets the session pixel format.
 *
 * This method causes a SetPixelFormat RFB message to be sent to a VNC server,
 * followed by a FramebufferUpdateRequest for the entire framebuffer.  The
 * VNC server will use the given pixel format for all future
 * FramebufferUpdates, until the next call to VNCViewerSetPixelFormat().
 *
 * If VNCViewerSetPixelFormat() is not called, then a VNC server will send
 * FramebufferUpdates in its native pixel format, which varies from server to
 * server.  You can explicitly select the server's native pixel format as the
 * session pixel format by calling VNCViewerSetPixelFormat() with pPixelFormat
 * equal to NULL.
 *
 * You may call this method before establishing a connection to a VNC server.
 * The SDK shall send the SetPixelFormat RFB message at the appropriate time
 * when establishing the VNC session.
 *
 * Some VNC servers may not support SetPixelFormat requests from the viewer in
 * certain circumstances. For example, MirrorLink servers do not support
 * changing pixel formats after a FramebufferUpdateRequest RFB message has been
 * sent by the viewer. You should therefore call this method before
 * establishing a connection to the server, if you wish to set the pixel format
 * for a MirrorLink VNC session. If the server does not support the
 * SetPixelFormat request, then the new pixel format shall only take effect
 * when a new VNC session is established. No indication is provided as to
 * whether or not the VNC server supports the SetPixelFormat RFB message at
 * this time.
 *
 * \param pViewer The VNC Viewer whose pixel format to set.
 * \param pPixelFormat If pPixelFormat is non-NULL, then it points to the new
 * session pixel format.  The SDK copies this data; it is safe to free this
 * structure after VNCViewerSetPixelFormat() returns.  If pPixelFormat is NULL,
 * then the new session pixel format is the server's native pixel format.
 *
 * \retval VNCViewerErrorNone The operation was successful.
 * \retval VNCViewerErrorInvalidParameter pPixelFormat is not NULL and
 * *pPixelFormat is not a supported pixel format.
 *
 * \see VNCPixelFormat, VNCViewerGetPixelFormat(),
 * VNCViewerGetActivePixelFormat()
 */
typedef VNCViewerError VNCCALL
VNCViewerSetPixelFormat(VNCViewer *pViewer,
                        const VNCPixelFormat *pPixelFormat);

/**
 * \brief Queries the chosen session pixel format.
 *
 * This method returns the last pixel format that was selected with
 * VNCViewerSetPixelFormat().  The structure pointed to by the returned
 * pointer is owned by the VNC Viewer; do not overwrite or free it, and
 * the pointer will become invalid when the VNC Viewer is destroyed. The
 * contents of the structure will be overwritten by subsequent calls to
 * VNCViewerGetPixelFormat().
 *
 * If VNCViewerSetPixelFormat() has not been called for this viewer instance,
 * then the session pixel format is the server's native pixel format, which
 * varies from server to server.  In this case, VNCViewerGetPixelFormat()
 * returns NULL.  The server's native pixel format for a particular session is
 * provided in the VNCViewerServerInitCallback.
 *
 * Some VNC servers may not support SetPixelFormat requests from the viewer in
 * certain circumstances. If VNCViewerSetPixelFormat() was called, and returned
 * VNCViewerErrorNone, then VNCViewerGetPixelFormat() will return the pixel
 * format that was specified in the call, even if the VNC server did not
 * support the format requested.
 *
 * \param pViewer The VNC Viewer whose session pixel format to query.
 *
 * \return The last selected session pixel format, or NULL if
 * VNCViewerSetPixelFormat() has not been called.
 *
 * \see ::VNCPixelFormat, VNCViewerSetPixelFormat(),
 * VNCViewerGetActivePixelFormat(), VNCViewerServerInitCallback()
 */
typedef const VNCPixelFormat *VNCCALL
VNCViewerGetPixelFormat(VNCViewer *pViewer);

/**
 * \brief Queries the pixel format currently in use.
 *
 * This method returns the pixel format currently in use for transmitting
 * pixel data from the server.  The structure pointed to by the returned
 * pointer is owned by the VNC Viewer; do not overwrite or free it, and
 * the pointer will become invalid when the VNC Viewer is destroyed. The
 * contents of the structure will be overwritten by subsequent calls to
 * VNCViewerGetActivePixelFormat().
 *
 * Note that this may be different from the chosen session pixel format
 * returned by VNCViewerGetPixelFormat() under some circumstances:
 * - If an attempt was made to set the pixel format using
 * VNCViewerSetPixelFormat() during a session which does not support pixel
 * mid-session format changes, e.g. MirrorLink, then the previous pixel
 * format will be returned as it will be still in use.
 * - If the server native pixel format is in use, the actual server pixel
 * format will be returned rather than NULL.
 * - If no pixel format is currently in use, i.e. the VNC Viewer is not
 * connected, or the connection has not yet established a pixel format,
 * then NULL will be returned rather than the pixel format configured using
 * VNCViewerSetPixelFormat().
 *
 * Also note that the framebuffer will continue to use the pixel format
 * specified when the VNC Viewer was created, for the lifetime of the VNC
 * Viewer, irrespective of the format used for transmitting data.
 *
 * \param pViewer The VNC Viewer whose currently active pixel format is to
 * be queried.
 *
 * \return The currently active pixel format, or NULL if no pixel format is
 * currently in use.
 *
 * \see ::VNCPixelFormat, VNCViewerSetPixelFormat(),
 * VNCViewerGetPixelFormat()
 */
typedef const VNCPixelFormat *VNCCALL
VNCViewerGetActivePixelFormat(VNCViewer *pViewer);

/**
 * \brief Optionally provide information describing the alpha channel present in
 * the framebuffer.
 *
 * By default the framebuffer is assumed to lack an alpha channel. When a
 * locked region of the framebuffer is updated, and an alpha channel is present,
 * it will be overwritten.
 *
 * An alpha channel may only be specified if the framebuffer pixel format is a
 * true color pixel format.
 *
 * \param pViewer The alpha mask is specified for the framebuffer associated
 * with this viewer.
 *
 * \param alphaMax The maximum value for the alpha component is
 * \code ((2 ^ n) - 1) \endcode, where \code n \endcode is an integer
 *
 * \param alphaShift The number of right shift operations required such that
 * the least significant bit of the alpha component coincides with the least
 * significant bit of an integer type of sufficient width to accomodate the
 * pixel components. The shifts are applied after transforming the pixel
 * representation to account for the native endianess of the current platform.
 *
 * \note The alpha component of a pixel can be extracted from a pixel value as
 * \code ((pixelValue >> alphaShift) & alphaMax) \endcode
 *
 * \note The alpha mask specified here will be used to convert the pixel format
 * of any VNCCursor passed to any callback registered as a
 * VNCViewerCursorUpdateCallback into the current framebuffer pixel format.
 * Since a VNCCursor requires a pixel format with an alpha channel, the callback
 * will not be invoked unless an alpha mask is specified by calling this method.
 *
 * \retval VNCViewerErrorNone The framebuffer alpha mask was successfully set
 * \retval VNCViewerErrorInvalidParameter
 * - pViewer is NULL or
 * - the current framebuffer pixel format is not true color or
 * - part or all of the alpha component for each pixel lies outside of the
 * bounds specified by the bitsPerPixel field of the current framebuffer pixel
 * format - parts or all of the alpha component overlap with parts or all of one
 * or more of the other three components of the current framebuffer pixel
 * format.
 *
 * \see VNCPixelFormat, VNCCursor, VNCViewerCursorUpdateCallback
 */
typedef VNCViewerError VNCCALL
VNCViewerSetFrameBufferAlphaMask(VNCViewer *pViewer,
                                 vnc_uint16_t alphaMax,
                                 vnc_uint8_t alphaShift);

/**
 * \brief Causes a KeyEvent RFB message to be sent to the VNC server.
 *
 * Call this method when the user presses or releases a key, so that the key
 * event can be forwarded to the VNC server.
 *
 * However, under Windows, you should use VNCViewerProcessWindowsKeyMessage()
 * instead of calling VNCViewerSendKeyEvent() directly.
 *
 * \param pViewer The VNC Viewer that should send the KeyEvent.
 * \param keyDown If non-zero, send a key-down event.  If zero, send a key-up
 * event.
 * \param keySymbol The X key symbol (e.g. XK_Tab) representing the key that
 * has been pressed or released.  These constants are defined in keysymdef.h.
 * To translate from a platform-native key symbol constant or a Unicode
 * character, call VNCSpecialKeyToXKeySymbol() or VNCUnicodeToXKeySymbol().
 * You should use VNCSpecialKeyToXKeySymbol() for keys that do not generate
 * characters (e.g. Shift) and VNCUnicodeToXKeySymbol() for keys that have
 * generated characters (your application must be able to differentiate between
 * such keypresses).
 *
 * \note Remember to send key up events that exactly match the key down events
 * that you have sent.  Failing to do so will cause the VNC server to act as
 * though the key is being held down.
 *
 * \note Not all platforms generate key up events for keys that were pressed
 * when the application loses the input focus.  However, your application
 * should still send RFB key up events for these keys.
 *
 * \see VNCSpecialKeyToXKeySymbol, VNCUnicodeToXKeySymbol, keysymdef.h
 */
typedef void VNCCALL
VNCViewerSendKeyEvent(VNCViewer *pViewer,
                      vnc_uint8_t keyDown,
                      vnc_int32_t keySymbol);

/**
 * \brief Causes a PointerEvent RFB message to be sent to the VNC server.
 *
 * You should call this method whenever the user moves the local mouse pointer
 * or changes its button state, provided that the pointer is over the area of
 * the screen in which your application is displaying the framebuffer.  This
 * will cause the pointer event to be sent to the VNC Server.
 *
 * Each call to VNCViewerSendPointerEvent() informs the VNC Server of the
 * pointer's current state.  The VNC Server maintains the pointer in this state
 * until the either the next call to VNCViewerSendPointerEvent() or the end of
 * the session.  Therefore, if you want to cause the server to click and then
 * release a mouse button, you must make two calls to
 * VNCViewerSendPointerEvent().
 *
 * For example, to perform a drag-and-drop from (0, 0) to (10, 10):
 *
 * \code
 * size_t i;
 *
 * // move the pointer with the left mouse button held down
 * for (i = 0; i < 10; i++)
 *     VNCViewerSendPointerEvent(pViewer, VNCPointerDeviceButtonLeft, i, i);
 *
 * // release the left mouse button at the final location
 * VNCViewerSendPointerEvent(pViewer, VNCPointerDeviceButtonNone, i, i);
 * \endcode
 *
 * \param pViewer The VNC Viewer that should send the PointerEvent.
 * \param buttonMask A bit-mask representing the pointer device buttons that
 * are currently depressed.  Use the VNCPointerDeviceButton enumeration to set
 * the bits in buttonMask.
 * \param x The x co-ordinate of the pointer on the framebuffer.
 * \param y The y co-ordinate of the pointer on the framebuffer.
 *
 * \see VNCPointerDeviceButton
 */
typedef void VNCCALL
VNCViewerSendPointerEvent(VNCViewer *pViewer,
                          VNCPointerDeviceButton buttonMask,
                          vnc_uint16_t x,
                          vnc_uint16_t y);

/**
 * \brief Sends a touch event to the VNC Server.
 *
 * Touch events differ from pointer events in that they encapsulate multiple
 * locations (usually of fingertips) with different pressures, whereas a
 * pointer event has a single point of contact (usually of a mouse pointer)
 * with a combination of different buttons.
 *
 * \note Not all server host machines support the injection of touch events.
 * Calling VNCViewerSendTouchEvent() when the server does not support touch
 * events will terminate the session with a ::VNCViewerErrorNoProtocolSupport
 * error.
 *
 * \note It is therefore recommended that, if you intend for your viewer
 * application to support touch events, you query
 * ::VNCViewerPropertyServerSupportsTouchEvents once, after the session has
 * been established, and keep a note of the result.
 *
 * A touch event is described by an array of VNCTouchDescriptor structures.
 * Each structure gives the location of a single point of contact, together
 * with the pressure applied and an integer identifier that allows the server
 * to track that particular point of contact as it moves.
 *
 * Your application must ensure that, when a point of contact is lost,
 * VNCViewerSendTouchEvent() is called again with a pressure of zero for that
 * point of contact.
 *
 * Every time you call this API you must include all points of contact that
 * belong to the current gesture, even if only some of them have changed state
 * since the last call.
 *
 * For example, to programmatically perform a pinch-zoom gesture, with one finger
 * moving from (5, 5) to (0, 0) and the other finger moving from (10, 10) to
 * (15, 15), the sequence of calls would be:
 *
 * \code
 * VNCTouchDescriptor touchDescriptors[2];
 * size_t x0, x1;
 *
 * touchDescriptors[0].pressure = 255;
 * touchDescriptors[0].identifier = 0;
 * touchDescriptors[1].pressure = 255;
 * touchDescriptors[1].identifier = 1;
 *
 * // apply both points of contact and move them to their final locations
 * for (x0 = 5, x1 = 10; x0 >= 0; x0--, x1++)
 * {
 *     touchDescriptors[0].location.x = x0;
 *     touchDescriptors[0].location.y = x0;
 *     touchDescriptors[1].location.x = x1;
 *     touchDescriptors[1].location.y = x1;
 *
 *     VNCViewerSendTouchEvent(pViewer, touchDescriptors, 2, 0, 0);
 * }
 *
 * // release both points of contact at their final locations
 * touchDescriptors[0].pressure = 0;
 * touchDescriptors[1].pressure = 0;
 * VNCViewerSendTouchEvent(pViewer, touchDescriptors, 2, 0, 0);
 * \endcode
 *
 * Your application should also ensure that it calls VNCViewerLostFocus()
 * whenever the viewer application loses focus. This will automatically
 * communicate to the server that an ongoing series of touch events has been
 * cancelled (some server platforms regard 'cancelled' and 'finished' as
 * separate outcomes for gesture recognition).
 *
 * If you want to use the timestamps provided by the touch-screen then you
 * can pass them to the SDK through the timestampSeconds and
 * timestampNanoseconds parameters. These values should represent the absolute
 * time at which the events were recorded. If instead you want the SDK to
 * handle the timestamps then simply set the two parameters to 0.
 *
 * A maximum of 255 concurrent touches are supported by this API. The
 * touchDescriptorCount should always be less than or equal to this limit.
 *
 * \param pViewer The VNC Viewer that should send the touch event.
 * \param touchDescriptors Array of VNCTouchDescriptors that specify the touch
 * event.
 * \param touchDescriptorCount The number of VNCTouchDescriptors in the
 * touchDescriptors array.
 * \param timestampSeconds The timestamp of the event in seconds.
 * \param timestampNanoseconds The timestamp of the event in nanoseconds.
 *
 * \see VNCViewerLostFocus
 */
typedef void VNCCALL
VNCViewerSendTouchEvent(VNCViewer *pViewer,
                        const VNCTouchDescriptor *touchDescriptors,
                        size_t touchDescriptorCount,
                        vnc_uint32_t timestampSeconds,
                        vnc_uint32_t timestampNanoseconds);

/**
 * \brief Generates an RSA public and private key pair for use by a VNC Viewer.
 *
 * Generating an RSA key pair and then assigning it to a VNC Viewer's
 * VNCParameterRSAKeys parameter allows it to use the RA2 family of VNC
 * security types.  It is recommended that you always perform this step, unless
 * you know in advance that your VNC session will not require encryption or
 * authentication.
 *
 * Generating an RSA key pair is a CPU-intensive operation and could take
 * some time.  You may wish to perform this operation in a background
 * thread to prevent your UI from becoming unresponsive, and to warn the user
 * that it may take a while.
 *
 * However, it is not necessary to generate a new RSA key pair for every
 * session.  You can perform this step during initial application setup or at
 * first use, and then securely store the key pair for future use.
 *
 * \param keyLength The length of the generated RSA keys, in bits.  This number
 * must be at least 512.  If keyLength is 0, the default key length will be
 * used.  The default key length is 2048 bits.
 *
 * \return Returns the new RSA key pair, formatted as a hexadecimal string.
 * When you are finished with this string, call VNCFreeString(), which will
 * zero the memory for you.  Returns NULL if an RSA key pair cannot be
 * generated.
 *
 * \note The private key component of the returned string is not encrypted.  If
 * you store it persistently (e.g. on disk or in the Windows registry), take
 * care to encrypt it first, or to restrict access to it appropriately.
 *
 * \see VNCParameterRSAKeys, VNCSecurityType
 */
typedef char *VNCCALL
VNCGenerateRSAKeys(size_t keyLength);

/**
 * \brief Causes a DeviceKey RFB Device extension message to be sent to the VNC
 * server.
 *
 * The RFB Device extension is a protocol extension that has been designed
 * specifically for interaction with VNC Servers on mobile devices.  The
 * DeviceKey message provided by this extension causes the VNC Server to inject
 * a press or release of a device hardware button.  These buttons fall into two
 * classes:
 *
 * - Raw device buttons.  These are button codes as defined by the server OS.
 *   When a VNC Server on a mobile device receives one of these, it will inject
 *   it into the OS without modification.
 * - Abstracted buttons.  VNC Mobile Solution defines abstractions for buttons
 *   that are commonly found on mobile devices, such as send and end, softkeys,
 *   cursor movement buttons, select, and so on.  When a VNC Server on a mobile
 *   device receives one of these, it will translate it into a raw device
 *   button code and then inject it into the OS.
 *
 * If the server is a MirrorLink server, then abstracted button codes are
 * mapped to the appropriate MirrorLink key symbol, and sent to the server as a
 * regular KeyEvent.
 *
 * Calling this method when the server is neither a VNC Mobile Solution server
 * nor a MirrorLink server has no effect.
 *
 * \param pViewer The VNC Viewer that should send the DeviceKey message.
 * \param raw If non-zero, then keyCode is a raw device button code, to be
 * interpreted by the server host OS.  Otherwise, it is a member of the
 * VNCDeviceKey enumeration.
 * \param keyCode The abstracted button code or raw device button code.
 * \param pressed If non-zero, then the VNC Server will inject a press of this
 * button.  Otherwise, the VNC Server will inject a release of this button.
 *
 * \see VNCDeviceKey
 */
typedef void VNCCALL
VNCViewerSendDeviceKeyEvent(VNCViewer *pViewer,
                            vnc_uint8_t raw,
                            vnc_uint32_t keyCode,
                            vnc_uint8_t pressed);

/**
 * \brief Causes a key to be sent to the RFB server either as a device key
 * event or as a normal key event, whichever is appropriate.
 *
 * The RFB Device extension is a protocol extension that has been designed
 * specifically for interaction with VNC Servers on mobile devices.  The
 * DeviceKey message provided by this extension causes the VNC Server to inject
 * a press or release of a device hardware button.
 *
 * Some keys should be delivered using that Device extension, whilst others
 * (such as normal ASCII text keys) should be delivered using the normal
 * RFB key event mechanism.
 *
 * This function attempts to do the 'best of both worlds'. If the key is
 * understood by the Device Extension, it will be sent that way. Otherwise, it
 * will be sent as a normal key event.  For example, if \a keySymbol is
 * \c XK_Home, \ref VNCDeviceKeyHome will be sent to the server as a device
 * key.  For most viewer applications, this will provide good behavior for the
 * end-user. More precise control can be obtained by calling
 * VNCViewerSendDeviceKeyEvent and VNCViewerSendKeyEvent instead.
 *
 * \note Because MirrorLink specifies that key mappings are done on the
 * server side, if connected to a MirrorLink server, the key event will be
 * sent through without any mapping. (See  the MirrorLink VNC Specification
 * (CCC-TS-041 v1.0, CCC-TS-010 v1.1, v1.2 and v1.3), "Event Mapping Messages")
 *
 * \param keyDown If non-zero, send a key-down event.  If zero, send a key-up
 * event.
 * \param keySymbol The X key symbol (e.g. XK_Tab) representing the key that
 * has been pressed or released.  These constants are defined in keysymdef.h.
 * To translate from a platform-native key symbol constant or a Unicode
 * character, call VNCSpecialKeyToXKeySymbol() or VNCUnicodeToXKeySymbol().
 * You should use VNCSpecialKeyToXKeySymbol() for keys that do not generate
 * characters (e.g. Shift) and VNCUnicodeToXKeySymbol() for keys that have
 * generated characters (your application must be able to differentiate between
 * such keypresses).
 * \see VNCViewerSendKeyEvent, VNCViewerSendDeviceKeyEvent
 */
typedef void VNCCALL
VNCViewerSendDeviceKeyOrKeyEvent(VNCViewer *pViewer,
                                 vnc_uint8_t keyDown,
                                 vnc_int32_t keySymbol);

/**
 * \brief Allocates a string for use by the SDK.
 *
 * Use this method to allocate strings whose ownership must be passed to the
 * SDK.
 *
 * \param string A string to copy.  See the note below.
 * \param length The length of the string to allocate.  See the note below.
 *
 * \return Returns a new string.  The returned string must be freed by calling
 * VNCFreeString().
 *
 * \note There are three ways to use this function:
 *
 *  - string != NULL, length == (size_t) -1 - copies an existing NUL-terminated
 *    string.
 *  - string != NULL, length >= 0 - copies an existing string of that length
 *    and places a NUL-terminator afterwards.
 *  - string == NULL, length >= 0 - allocates a buffer that can hold a string
 *    of that length and also a NUL-terminator (i.e. it allocates (length + 1)
 *    bytes).  The entire buffer is initialized with NULs.
 */
typedef char *VNCCALL
VNCAllocString(const char *string, size_t length);

/**
 * \brief Frees a string that has been allocated by the SDK or by calling
 * VNCAllocString().
 *
 * Use this method to free any strings that have been allocated in this way but
 * are no longer required.  The memory is zeroed prior to being freed.
 *
 * The provided string must be NUL terminated if non-NULL.
 *
 * It is guaranteed to be safe to call VNCFreeString() with a NULL parameter.
 *
 * \param string The string to free.
 */
typedef void VNCCALL
VNCFreeString(char *string);

/**
 * \brief Translates a platform-specific special key symbol into an X key
 * symbol.
 *
 * Special keys are keys that do not produce characters and are not dead keys.
 * These include Shift, Control, IME mode change keys, and so on.
 *
 * On Windows, the platform-specific key symbols are the virtual key (VK_*)
 * constants defined in winuser.h (or, equivalently, members of the .Net
 * Framework's System.Windows.Forms.Keys enumeration).  On Linux,
 * the platform-specific key symbols are the X key symbols, and this function
 * is a no-op. On QNX and Android this operation will also be a no-op, which
 * may not be applicable depending on the GUI framework in use.
 *
 * \param platformSpecificKeySymbol The platform-specific key symbol.
 *
 * \return The X key symbol corresponding to platformSpecificKeySymbol, or
 * (vnc_uint32_t) -1 if this special key is not supported by VNC.
 *
 * \see VNCViewerSendKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCSpecialKeyToXKeySymbol(vnc_uint32_t platformSpecificKeySymbol);

/**
 * \brief Translates an X key symbol into a platform-specific special key
 * symbol.
 *
 * Special keys are keys that do not produce characters and are not dead keys.
 * These include Shift, Control, IME mode change keys, and so on.
 *
 * On Windows, the platform-specific key symbols are the virtual key (VK_*)
 * constants defined in winuser.h (or, equivalently, members of the .Net
 * Framework's System.Windows.Forms.Keys enumeration).  On Linux,
 * the platform-specific key symbols are the X key symbols, and this function
 * is a no-op.
 *
 * \param keySymbol An X key symbol.
 *
 * \return The platform-specific special key symbol corresponding to keySymbol,
 * or (vnc_uint32_t) -1 if this X key symbol is not supported by VNC on this
 * platform.
 *
 * \see VNCViewerSendKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCXKeySymbolToSpecialKey(vnc_uint32_t keySymbol);

/**
 * \brief Translates a Unicode character into an X key symbol.
 *
 * The translation is independent of any VNC Viewer object, and therefore
 * independent of any VNC Server to which the SDK may have established a
 * session.
 *
 * \deprecated This API is deprecated in favour of
 * ::VNCViewerUnicodeToXKeySymbol(), which takes a VNC Viewer object as a
 * parameter, and can therefore perform a translation that is guaranteed to be
 * appropriate for the current VNC session.
 *
 * \param unicodeCharacter A Unicode character in native-endian UTF-32 encoding.
 *
 * \return The X key symbol corresponding to unicodeCharacter, or
 * (vnc_uint32_t) -1 if there is none.
 *
 * \see VNCViewerSendKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCUnicodeToXKeySymbol(vnc_uint32_t unicodeCharacter);

/**
 * \brief Translates a Unicode character into an X key symbol, taking account
 * of the type of the VNC Server to which the given VNC Viewer is connected.
 *
 * The translation is performed in such a way as to maximize interoperability
 * with the the VNC Server to which the given VNC Viewer object is connected.
 * For this reason, VNCViewerUnicodeToXKeySymbol() is preferred to the
 * deprecated ::VNCUnicodeToXKeySymbol(), which does not take a VNC Viewer
 * object as a parameter and therefore cannot adapt itself to particular VNC
 * Servers.
 *
 * If the given VNC Viewer is not connected to a VNC Server, or the type of VNC
 * Server has not yet been determined, then the translation performed by
 * VNCViewerUnicodeToXKeySymbol() is identical to that performed by
 * ::VNCUnicodeToXKeySymbol().
 *
 * \param pViewer The VNC Viewer for which the character should be translated.
 * \param unicodeCharacter A Unicode character in native-endian UTF-32 encoding.
 *
 * \return The X key symbol corresponding to unicodeCharacter, or
 * (vnc_uint32_t) -1 if there is none.
 *
 * \see VNCViewerSendKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCViewerUnicodeToXKeySymbol(VNCViewer *pViewer, vnc_uint32_t unicodeCharacter);

/**
 * \brief Translates an X key symbol into a Unicode character.
 *
 * \param keySymbol An X key symbol.
 *
 * \return A Unicode character, in native-endian UTF-32 encoding, corresponding to
 * keySymbol, or (vnc_uint32_t) -1 if there is none.
 *
 * \see VNCViewerSendKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCXKeySymbolToUnicode(vnc_uint32_t keySymbol);

/**
 * \brief Processes Windows key event messages and coverts them to RFB KeyEvent
 * messages.
 *
 * The recommended way for Windows applications to forward key events to a VNC
 * Server is to pass every WM_KEYDOWN, WM_KEYUP, WM_SYSKEYDOWN, WM_SYSKEYUP and
 * WM_CHAR message to VNCViewerProcessWindowsKeyMessage().  The keyboard
 * handling logic within the Viewer SDK deals with the complex issues of
 * keyboard shift state, international keyboard layouts and dead key
 * processing.
 *
 * When the VNC Server is a mobile device, VNCViewerProcessWindowsKeyMessage()
 * may optionally map certain keys to RFB Device extension DeviceKey messages.
 * This allows the end user to control the mobile device's buttons using the
 * local keyboard.  In this release, the key mapping is hard-coded.  Consult
 * the release documentation for details of this key mapping.
 *
 * If your application calls VNCViewerProcessWindowsKeyMessage(), you should
 * remember to call VNCViewerLostFocus() when your application loses the
 * input focus.
 *
 * \param pViewer The VNC Viewer that should send the KeyEvent.
 * \param msg The Windows message type.  This should be one of WM_KEYDOWN,
 * WM_KEYUP, WM_SYSKEYDOWN and WM_SYSKEYUP.
 * \param wparam The WPARAM from the Windows message.
 * \param lparam The LPARAM from the Windows message.
 * \param map If this is non-zero, and the VNC Server is a mobile device, and
 * the key that was pressed maps to a device key, then a DeviceKey message will
 * be sent to the server.  Otherwise, a KeyEvent message will be sent to the
 * server.
 *
 * \see VNCViewerLostFocus, VNCViewerSendKeyEvent,
 * VNCViewerSendDeviceKeyEvent
 */
typedef void VNCCALL
VNCViewerProcessWindowsKeyMessage(VNCViewer *pViewer,
                                  vnc_uint32_t message,
                                  vnc_uintptr_t wparam,
                                  vnc_intptr_t lparam,
                                  vnc_uint8_t  map);

/**
 * \brief Informs the SDK that the VNC Viewer has lost the input focus.
 *
 * Calling this method cancels any touch, pointer or key events currently
 * in progress. If the application has used any of these, then it must call
 * this method whenever the viewer UI loses the input focus.
 *
 * See VNCViewerSendTouchEvent() for a fuller explanation.
 *
 * \param pViewer The VNC Viewer that has lost the input focus.
 *
 * \see VNCViewerSendTouchEvent()
 */
typedef void VNCCALL
VNCViewerLostFocus(VNCViewer *pViewer);

/**
 * \brief 'Undoes' KeyEvent and DeviceKey messages that have been generated by
 * VNCViewerProcessWindowsKeyMessage().
 *
 * VNCViewerProcessWindowsKeyMessage() maintains the current keyboard state so
 * that it can generate KeyEvent and DeviceKey messages for key releases that
 * exactly correspond to those that it generated for key presses.  The effect
 * of calling VNCViewerReleaseAllKeys() is to send all of these pending key
 * release messages at once, and then reset the keyboard state so that no
 * key releases are pending.
 *
 * \deprecated This API is deprecated in favour of ::VNCViewerLostFocus(),
 * which handles touch, pointer and key events.
 *
 * This method has no effect on platforms other than Windows.
 *
 * \param pViewer The VNC Viewer whose keyboard state should be reset.
 *
 * \see VNCViewerProcessWindowsKeyMessage
 */
typedef void VNCCALL
VNCViewerReleaseAllKeys(VNCViewer *pViewer);

/**
 * \brief Selects a mobile device skin for use with a VNC session.
 *
 * Mobile device skins allow for a graphical representation of the device being
 * remotely controlled to be displayed within the Viewer SDK's framebuffer.
 *
 * A skin contains 'skinface' images that depict the device in its various
 * physical configurations.  For example, if the device in question has a
 * two-position slide or a flip, then there should be one skinface image for
 * each of these configurations.  A skin also contains XML metadata that
 * describes the skin's behavior and defines the locations and shapes of the
 * buttons and other physical controls within the skinface images.
 *
 * From the perspective of the application, skinned and unskinned VNC sessions
 * behave identically.  Pointer location translations, skinface image rotation,
 * skin button simulation and input events generated by skin buttons are dealt
 * with internally to the Viewer SDK.  The only additional responsibilities for
 * the application are to decide which skin to use and to ensure that it is
 * available via the filesystem.  If skins are stored remotely, you can use one
 * or more VNCViewerSkinFetchCallback()s to download or otherwise make a local
 * copy of the appropriate skin.  You may also use
 * VNCViewerSkinFetchCallback() to decide at runtime which skin to use
 * according to the type of device that is hosting the VNC Server.
 *
 * You should then call VNCViewerSetSkin(), which provides the SDK with the
 * path to the skin archive.  After VNCViewerSetSkin() returns, the Viewer SDK
 * will not access the skin archive file again.  So, you are free to remove it
 * if you wish to do so.
 *
 * By default, VNC sessions are not skinned.  To return to unskinned behavior
 * after previously having selected a skin, call VNCViewerSetSkin() with NULL
 * as the skin archive filename parameter.
 *
 * Calls to VNCViewerSetSkin() take effect immediately, whether or not the
 * viewer thread is running.
 *
 * This method will fail if a pluggable renderer is loaded.
 *
 * \param pViewer The VNC Viewer whose skin is being selected.
 * \param skin If non-NULL, this is the name of a file which is a zip archive
 * containing the mobile device skin.  If NULL, this indicates that no skin
 * should be used.
 * \param reserved Reserved for future use.  The value of this parameter may be
 * used in a future release to indicate the skin archive container format.
 * Currently, the only supported container format is a zip archive, so the
 * value of this parameter is ignored.  However, for future compatibility, the
 * value of this parameter should be zero.
 *
 * \return VNCViewerErrorNone if the skin was successfully processed.
 *
 * \see VNCViewerSkinFetchCallback, VNCParameterSkinAlignment,
 * VNCParameterSkinBackgroundColor, VNCParameterSkinButtonHoverHighlight,
 * VNCParameterSkinButtonPressHighlight
 */
typedef VNCViewerError VNCCALL
VNCViewerSetSkin(VNCViewer *pViewer,
                 const char *filename,
                 vnc_uintptr_t reserved);

/**
 * \brief Registers a VNCViewerSkinFetchCallback() with a VNC Viewer object.
 *
 * You can use the pSkinFetchCallbackContext parameter to provide your callback
 * with configuration or state information.  This pointer allows the
 * application to register the same callback more than once with multiple
 * different configurations.
 *
 * Note that the SDK does not manage the memory pointed to by
 * pSkinFetchCallbackContext.  It is the responsibility of the application to
 * ensure that this pointer remains valid until either the callback is
 * unregistered or the VNC Viewer object is destroyed.
 *
 * \param pViewer The VNC Viewer for which the callback should be registered.
 * \param pSkinFetchCallback The address of the callback.
 * \param pSkinFetchCallbackContext A context pointer that is particular to
 * this registration of this callback.
 *
 * \return A key that can be used to identify the callback to
 * VNCViewerRemoveSkinFetchCallback().
 *
 * \see VNCViewerSkinFetchCallback(), VNCViewerRemoveSkinFetchCallback(),
 * VNCViewerRemoveAllSkinFetchCallbacks()
 */
typedef vnc_uint32_t VNCCALL
VNCViewerAddSkinFetchCallback(VNCViewer *pViewer,
                              VNCViewerSkinFetchCallback *pSkinFetchCallback,
                              void *pSkinFetchCallbackContext);

/**
 * \brief Unregisters a VNCViewerSkinFetchCallback() from a VNC Viewer object.
 *
 * \param pViewer The VNC Viewer from which the callback should be
 * unregistered.
 * \param skinFetchCallbackId The key returned by
 * VNCViewerAddSkinFetchCallback() when the callback was registered.
 *
 * \see VNCViewerSkinFetchCallback(), VNCViewerAddSkinFetchCallback(),
 * VNCViewerRemoveSkinFetchCallback(),
 */
typedef void VNCCALL
VNCViewerRemoveSkinFetchCallback(VNCViewer *pViewer,
                                 vnc_uint32_t skinFetchCallbackId);

/**
 * \brief Unregisters all VNCViewerSkinFetchCallback()s from a VNC Viewer
 * object.
 *
 * \param pViewer The VNC Viewer from which the callbacks should be
 * unregistered.
 *
 * \see VNCViewerSkinFetchCallback(), VNCViewerAllSkinFetchCallback(),
 * VNCViewerRemoveSkinFetchCallback(),
 */
typedef void VNCCALL
VNCViewerRemoveAllSkinFetchCallbacks(VNCViewer *pViewer);

/**
 * \brief Notifies the Viewer SDK that the pending VNCViewerSkinFetchCallback()
 * has completed.
 *
 * VNCViewerSkinFetchCallback()s are expected to execute asynchronously.  This
 * method exists to notify the SDK that the last callback that the SDK invoked
 * should now be considered complete.
 *
 * If the skinFetchingShouldStop parameter is zero, and there are any further
 * VNCViewerSkinFetchCallback()s remaining to be invoked, then the SDK will
 * schedule the next one for invocation.  (The callback will not be invoked
 * directly from VNCViewerSkinFetchCallbackComplete(), so your application
 * should not have to concern itself with re-entrancy issues.)
 *
 * If the skinFetchingShouldStop parameter is non-zero, no further
 * VNCViewerSkinFetchCallback()s will be invoked for the remainder of the
 * current VNC session.  In this case, it is expected that the
 * VNCViewerSkinFetchCallback() will call VNCViewerSetSkin() to apply the
 * results of the skin fetch before it calls
 * VNCViewerSkinFetchCallbackComplete().
 *
 * \param pViewer The VNC Viewer whose VNCViewerSkinFetchCallback has
 * completed.
 * \param skinFetchingShouldStop If true, then no more
 * VNCViewerSkinFetchCallback()s will be invoked.  Otherwise, skin fetching
 * continues with the next VNCViewerSkinFetchCallback.
 *
 * \see VNCViewerSkinFetchCallback()
 */
typedef void VNCCALL
VNCViewerSkinFetchCallbackComplete(VNCViewer *pViewer,
                                   vnc_int32_t skinFetchingShouldStop);

/**
 * \brief Causes a ClientCutText RFB message to be sent to the VNC server.
 *
 * If your application is running on a platform that supports a clipboard, and
 * you are able to observe the clipboard's contents, then you may wish to call
 * this method when the clipboard has new text copied to it.  This will cause
 * the VNC server host's clipboard contents to be updated to match the new
 * text, allowing the user to copy text from a local application to the
 * framebuffer.
 *
 * The text sent to the server can be truncated before sending, depending on
 * the value of the VNCParameterMaxCutText parameter.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param clipboardText The text to be copied to the VNC server host's
 * clipboard.  This should be UTF-8 encoded, but need not be NUL-terminated.
 * \param clipboardTextLength The length of clipboardText in bytes.
 *
 * \see VNCViewerServerCutTextCallback, VNCParameterMaxCutText
 */
typedef void VNCCALL
VNCViewerSendClientCutText(VNCViewer *pViewer,
                           const char *clipboardText,
                           size_t clipboardTextLength);

/**
 * \brief Causes a FramebufferUpdateRequest RFB message to the sent to the VNC
 * server.
 *
 * This method exists to give an application fine control over the process of
 * refreshing the framebuffer to match the server display.  By default, the SDK
 * requests framebuffer updates automatically; most applications will not need
 * this method.
 *
 * A suggested pattern for using this method is:
 *
 * -# Before connecting to the VNC Server, disable automatic update requests by
 *    calling VNCViewerEnableDisplayUpdates(FALSE).
 * -# When the VNCViewerSessionProgressCallback is received with a value of
 *    VNCViewerSessionProgressSessionEstablished, call
 *    VNCViewerFrameBufferUpdateRequest(NULL, FALSE) to request the initial
 *    update of the entire framebuffer.
 * -# To synchronize the framebuffer with the framebuffer, call
 *    VNCViewerFrameBufferUpdateRequest(NULL, TRUE) to update only those parts
 *    of the framebuffer that are out of date.
 *
 * \param pViewer The VNC Viewer that should send the FramebufferUpdateRequest.
 * \param pRectangle Optionally specifies the rectangle within the framebuffer
 * that should be updated.  If pRectangle == NULL, the entire framebuffer will
 * be updated (subject to the incremental flag).
 * \param incremental If incremental != 0, the request issued is incremental.
 * An incremental update contains only those parts of the rectangle that
 * have changed since the last time the rectangle was updated.  If
 * incremental == 0, the server will update the entire rectangle.
 */
typedef void VNCCALL
VNCViewerFrameBufferUpdateRequest(VNCViewer *pViewer,
                                  const VNCRectangle *pRectangle,
                                  vnc_int32_t incremental);

/**
 * \brief Withdraws a FramebufferUpdateRequest that was previously sent to the
 * VNC server.
 *
 * This method exists to give an application fine control over the process of
 * refreshing the framebuffer to match the server display.  By default, the SDK
 * requests framebuffer updates automatically; most applications will not need
 * this method.
 *
 * \param pViewer The VNC Viewer that should send the FramebufferUpdateRequest.
 * \param pRectangle Optionally specifies the rectangle within the framebuffer
 * for which an update request should be cancelled.  If pRectangle == NULL,
 * update requests for the entire framebuffer will be cancelled.
 */
typedef void VNCCALL
VNCViewerFrameBufferUpdateCancel(VNCViewer *pViewer,
                                 const VNCRectangle *pRectangle);

/**
 * \brief Notifies the viewer SDK that the viewer application has rendered the
 * framebuffer.
 *
 * This should be invoked by the viewer application at the end of the paint
 * method that redraws the server display image. It is assumed that all
 * invalidated regions of the framebuffer identified by prior
 * \ref VNCViewerUnlockRectangleExCallback
 * or
 * \ref VNCViewerUnlockRectangleExCallback
 * invocations shall have been redrawn.
 *
 * Providing this notification will assist the SDK in performance monitoring
 * and improvement.
 */
typedef void VNCCALL
VNCViewerFrameBufferRendered(VNCViewer *pViewer);

/**
 * \brief Notifies the Viewer SDK of the result of a
 * VNCViewerAsynchronousCredentialsCallback.
 *
 * Your application must call this method to notify the Viewer SDK of the
 * result of your VNCViewerAsynchronousCredentialsCallback().  This call may be
 * made from within your VNCViewerAsynchronousCredentialsCallback, or at a
 * later time, and from any thread.
 *
 * Unsolicited calls to VNCViewerAsynchronousCredentialsResult() are ignored.
 *
 * \param pViewer The VNC Viewer to which the call applies.
 * \param result Zero if your application was able to successfully
 * determine the credentials.  Non-zero indicates that credentials could not be
 * determined (e.g. if you displayed UI to gather the credentials and the user
 * cancelled it).  In this case, the username and password parameters are
 * ignored.
 * \param username The username for the session, as a NUL-terminated UTF-8
 * string. It gets truncated to 255 bytes before being sent, so it is
 * recommended to ensure the parameter does not exceed that length. If the
 * VNCViewerAsynchronousCredentialsCallback indicated that a username is not
 * required, then this parameter is ignored and may be NULL.
 * \param password The password for the session, as a NUL-terminated UTF-8
 * string. It gets truncated to 255 bytes before being sent, so it is
 * recommended to ensure the parameter does not exceed that length. If the
 * VNCViewerAsynchronousCredentialsCallback indicated that a
 * password is not required, then this parameter is ignored and may be NULL.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks,
 * VNCViewerAsynchronousCredentialsCallback
 */
typedef void VNCCALL
VNCViewerAsynchronousCredentialsResult(VNCViewer *pViewer,
                                       vnc_int32_t result,
                                       const char *username,
                                       const char *password);

/**
 * \brief Notifies the Viewer SDK of the result of a
 * VNCViewerAsynchronousValidatePasswordCallback.
 *
 * Your application must call this method to notify the Viewer SDK of the
 * result of your VNCViewerAsynchronousValidatePasswordCallback().  This call
 * may be made from within your VNCViewerAsynchronousValidatePasswordCallback,
 * or at a later time, and from any thread.
 *
 * Unsolicited calls to VNCViewerAsynchronousValidatePasswordResult() are
 * ignored.
 *
 * \param pViewer The VNC Viewer to which the call applies.
 * \param success Non-zero if the password is acceptable to your application,
 * or zero if not.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks,
 * VNCViewerAsynchronousValidatePasswordCallback
 */
typedef void VNCCALL
VNCViewerAsynchronousValidatePasswordResult(VNCViewer *pViewer,
                                            vnc_int32_t success);

/**
 * \brief Notifies the Viewer SDK of the result of a
 * VNCViewerAsynchronousVerifyServerIdentityCallback.
 *
 * Your application must call this method to notify the Viewer SDK of the
 * result of your VNCViewerAsynchronousValidatePasswordCallback().  This call
 * may be made from within your
 * VNCViewerAsynchronousVerifyServerIdentityCallback, or at a later time, and
 * from any thread.
 *
 * Unsolicited calls to VNCViewerAsynchronousVerifyServerIdentityResult() are
 * ignored.
 *
 * \param pViewer The VNC Viewer to which the call applies.
 * \param success Non-zero if the signature is acceptable to your application,
 * or zero if not.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks,
 * VNCViewerAsynchronousVerifyServerIdentityCallback
 */
typedef void VNCCALL
VNCViewerAsynchronousVerifyServerIdentityResult(VNCViewer *pViewer,
                                                vnc_int32_t success);

/**
 * \brief Registers an application-defined protocol extension with the Viewer
 * SDK.
 *
 * Protocol extensions allow the application to extend the RFB protocol with
 * application-defined messages.  These messages allow the application to
 * exchange arbitrary data with a compatible VNC Server application.
 *
 * Each protocol extension must be given a unique name.  A legal extension name
 * must:
 *
 *  - be a nonempty NUL-terminated string
 *  - have length less than or equal to 255 characters
 *  - consist only of the ASCII characters '0'-'9', 'A'-'Z', 'a'-'z', '_' and
 *    '.'
 *  - contain at least one '.'
 *
 * To provide namespacing, you should choose extension names so that they
 * follow the 'reverse domain name' precedent set by Java packages (see
 * http://en.wikipedia.org/wiki/Java_package#Package_naming_conventions).  For
 * example, com.example.first and com.example.second would be good choices for
 * the names of two extensions defined by the organization owning the
 * example.com domain.
 *
 * VNC Mobile Servers support only up to eight application-defined extensions
 * in the each server application.  For this reason, if your application has
 * many different types of extension message to send, it is recommended that
 * you define a common message type field within your extension messages,
 * rather than registering a different extension for each message type.
 *
 * Extensions must be registered before the viewer thread is started.  Once
 * registered, they remain registered for the lifetime of the viewer object,
 * and cannot be unregistered.
 *
 * \param pViewer The VNC Viewer for which the extension should be registered.
 * \param extensionName The extension name.
 * \param pExtensionEnabledCallback The function to be called when the VNC
 * Server enables the extension.
 * \param pExtensionMessageReceivedCallback The function to be called when the
 * a message is received from the VNC Server on this extension.
 * \param ppExtension On successful return, *ppExtension is an opaque pointer to
 * the extension that has been registered.
 *
 * \retval VNCViewerErrorNone The extension was registered successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The extension could not be
 * registered because the viewer thread is running.
 * \retval VNCViewerErrorInvalidParameter The given extension name is illegal.
 * \retval VNCViewerErrorAlreadyExists An extension with this name is already
 * registered.
 *
 * \see VNCViewerSendExtensionMessage, VNCViewerExtensionEnabledCallback,
 * VNCViewerExtensionMessageReceivedCallback
 */
typedef VNCViewerError VNCCALL
VNCViewerRegisterExtension(
    VNCViewer *pViewer,
    const char *extensionName,
    VNCViewerExtensionEnabledCallback *pExtensionEnabledCallback,
    VNCViewerExtensionMessageReceivedCallback *pExtensionMessageReceivedCallback,
    VNCExtension **ppExtension);

/**
 * \brief Sends a message on an application-defined protocol extension.
 *
 * An extension must be both registered and enabled by the VNC Server before a
 * message can be sent or received on it.  To register an extension, call
 * VNCViewerRegisterExtension().
 *
 * The VNCViewerExtensionEnabledCallback() notifies your application when the
 * VNC Server has enabled the extension This usually takes place soon after the
 * session has been established.
 *
 * The maximum supported length of an extension message payload is 4096 bytes.
 *
 * The extension message payload is copied by the SDK.  The application may
 * choose to free or overwrite it as soon as VNCViewerSendExtensionMessage()
 * has returned.
 *
 * Messages sent on a single extension are guaranteed to be delivered in
 * the order that the calls to VNCViewerSendExtensionMessage() are made.
 * However, messages sent on multiple different extensions are <i>not</i>
 * guaranteed to be delivered in the order that the calls to
 * VNCViewerSendExtensionMessage() are made.
 *
 * \par Reliability
 *
 * A successful return from VNCViewerSendExtensionMessage() is a guarantee that
 * the message has been queued for sending, but this is <i>not</i> a guarantee
 * of successful delivery.  This is because the session may be lost before the
 * message is received by the peer.
 *
 * Applications that require a guarantee of successful delivery must implement
 * this themselves, e.g. by using acknowledgements.
 *
 * \par Update starvation
 *
 * Extension messages sent from server to viewer are competing for bandwidth
 * with framebuffer updates.  This means that, if an application sends a large
 * volume of extension message data from server to viewer, the server will have
 * less bandwidth available for updates, and the framerate will decrease.
 *
 * The converse is also true; if the server display updates rapidly, then the
 * resulting framebuffer updates will increase the time it takes to send
 * extension messages to the viewer.
 *
 * If your server application has a large volume of extension message data to
 * send to the viewer, then you may wish to limit the rate at which extension
 * messages are sent, so that the impact on framebuffer updates is lessened.
 *
 * Alternatively, if you wish to complete the extension message data transfer
 * as quickly as possible, then you may wish to disable updates entirely until
 * the transfer is complete, and display some alternative viewer-side UI
 * instead (such as a progress bar).
 *
 * \par Handling VNCViewerErrorInsufficientBufferSpace
 *
 * ::VNCViewerErrorInsufficientBufferSpace indicates that the SDK cannot queue
 * an extension message for sending due to a temporary resource shortage.  In
 * this case, the application can make another attempt to queue the message at
 * a later time.
 *
 * Two possible strategies for avoiding this condition, and handling it should
 * it arise, are:
 *
 *  - Measure the average rate at which extension message data can successfully
 *    be queued in your deployment.  Queue extension messages at or just below
 *    that rate, and, if an attempt to queue a message fails, retry after a
 *    short timeout.
 *
 *  - Require the application receiving the extension messages to acknowledge
 *    each message.  Whenever you receive an acknowledgement for one message,
 *    attempt (or re-attempt) to queue the next one.  (You may wish to use a
 *    sliding window for acknowledgements, as in TCP.)
 *
 * \param pViewer The VNC Viewer that should send the extension message.
 * \param pExtension The VNCExtension pointer returned by
 * VNCViewerRegisterExtension().
 * \param payload The payload of the extension message.
 * \param payloadLength The length of the payload of the extension message.
 *
 * \retval VNCViewerErrorNone The message was queued successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorExtensionNotEnabled The VNC Server has not enabled
 * this extension.
 * \retval VNCViewerErrorInvalidParameter The payload length is too large.
 * \retval VNCViewerErrorNoProtocolSupport Extension messages are not supported
 * by the version of RFB that has been negotiated for the session.
 * \retval VNCViewerErrorInsufficientBufferSpace The SDK cannot queue the
 * message at this time (the application can retry later).
 *
 * \see VNCViewerRegisterExtension, VNCViewerExtensionEnabledCallback,
 * VNCViewerExtensionMessageReceivedCallback
 */
typedef VNCViewerError VNCCALL
VNCViewerSendExtensionMessage(VNCViewer *pViewer,
                              VNCExtension *pExtension,
                              const vnc_uint8_t *payload,
                              size_t payloadLength);

/**
 * \brief Associates a context pointer with an application-defined protocol
 * extension.
 *
 * Use this method to associate data with a VNCExtension instance so that you
 * can recover it at a later time. The context pointer is supplied to all
 * callbacks made by the SDK for that extension, and can also be retrieved by
 * calling VNCViewerGetExtensionContext().
 *
 * A second call to VNCViewerSetExtensionContext() replaces the context pointer
 * assigned by the first call.
 *
 * \param pViewer The viewer with which the extension was registered.
 * \param pExtension The extension with which to associate the context pointer.
 * \param pContext The context pointer to associate with the extension.
 *
 * \note The SDK will not interact in any way with the memory to which the
 * context pointer points.  In particular, the SDK will not free the memory
 * when the viewer is destroyed.
 *
 * \see VNCViewerGetExtensionContext
 */
typedef void VNCCALL
VNCViewerSetExtensionContext(VNCViewer *pViewer,
                             VNCExtension *pExtension,
                             void *pContext);

/**
 * \brief Retrieve the context pointer with an application-defined protocol
 * extension.
 *
 * A second call to VNCViewerSetExtensionContext() replaces the context pointer
 * assigned by the first call.
 *
 * \param pViewer The viewer with which the extension was registered.
 * \param pExtension The extension for which to retrieve the context pointer.
 * \return The context pointer associated with the extension, or NULL if there
 * is none.
 *
 * \see VNCViewerSetExtensionContext()
 */
typedef void *VNCCALL
VNCViewerGetExtensionContext(VNCViewer *pViewer,
                             VNCExtension *pExtension);

/**
 * \brief Translates from an encoding identifier to its textual representation.
 *
 * \param encoding The identifier for a VNC encoding type.
 *
 * \return SDK-owned string containing the encoding name, NULL if not
 * recognised.
 */
typedef const char *VNCCALL
VNCGetEncodingName(VNCEncoding encoding);

/**
 * \brief Adds a VNC license to the VNC Viewer.
 *
 * The Viewer SDK must be licensed in order to enable the use of certain
 * features and to connect to certain different types of VNC Servers. To obtain
 * a VNC license, contact your RealVNC sales representative.
 *
 * Calls to VNCViewerAddLicense() must be made before starting the viewer
 * thread.
 *
 * \param pViewer The VNC Viewer to which to add the license.
 * \param licenseText The text of the license. This is a NULL-terminated
 * string, and should contain the entire license including the header and
 * footer. If your license has been supplied to you in a .vnclicense file, then
 * licenseText should contain the entire contents of that file.
 * \param serialNumber If serialNumber is not NULL, then, on successful return,
 * it contains the license serial number, which is a big-endian UUID. If the
 * serial number is not required, then set serialNumber to NULL.
 * \param serialNumberSize The size in bytes of the buffer pointed to by
 * serialNumber. If serialNumber is not NULL, then serialNumberSize should be
 * at least 16. If the serialNumber is NULL, then serialNumberSize should be 0.
 *
 * \retval VNCViewerErrorNone The license was added successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The license cannot be added
 * because the viewer thread is already running.
 * \retval VNCViewerErrorInvalidParameter licenseText is NULL, or the
 * combination of serialNumber and serialNumberSize parameters is not valid.
 * \retval VNCViewerErrorLicenseNotValid licenseText does not contain a valid
 * VNC license.
 */
typedef VNCViewerError VNCCALL
VNCViewerAddLicense(VNCViewer *pViewer,
                    const char *licenseText,
                    vnc_uint8_t *serialNumber,
                    size_t serialNumberSize);

/**
 * \brief Performs a local feature check.
 *
 * This call returns true if any of the feature IDs supplied are
 * provided by a license previously added locally to this server
 * instance using addLicense().
 *
 * Note that user features added using provideLicenseFeature, and
 * features provided by a connected viewer, are not considered when
 * performing local feature checks.
 *
 * \param featureIDs The array of feature IDs.
 * \param featureIDCount The number of feature IDs in the array.
 * \param pResult If pResult is not NULL, then on successful return it will be
 * set to true if at least one of the feature IDs * is provided by a local
 * license, false otherwise.
 *
 * \retval VNCViewerErrorInvalidParameter, if the featureIds array is
 *   null or empty.
 */
typedef VNCViewerError VNCCALL
VNCViewerLocalFeatureCheck(VNCViewer *pViewer,
                           const vnc_uint32_t *featureIDs,
                           size_t featureIDCount,
                           vnc_uint8_t *pResult);

/**
 * \brief Defines a licensed feature on the VNC Viewer.
 *
 * A Viewer SDK application can define additional licensed features and then
 * challenge the VNC Server to provide evidence that it is licensed to use that
 * feature.  This functionality is typically used to restrict viewers and
 * servers so that they can connect only to viewers and servers authored by the
 * same vendor. This scenario is known as a <i>closed system</i>.
 *
 * For example, suppose that the VNC Viewer application:
 *
 *  - calls VNCViewerAddLicenseFeature() to define a new feature with ID X
 *  - calls VNCViewerAddRemoteFeatureCheck() to schedule a check that the VNC
 *    Server provides feature X
 *
 * In that case, for a session with a particular VNC Mobile Server to succeed,
 * that server application must:
 *
 *  - define feature X, with the same key that was provided in the viewer
 *  - provide feature X to answer the viewer's challenges
 *
 * (The VNC Mobile Server SDK APIs for these operations are named similarly and
 * to their Viewer SDK counterparts.)
 *
 * If the server does not provide feature X, or if the server defines feature X
 * with a feature key that differs from that defined in the viewer, then the
 * VNCViewerRemoteFeatureCheckFailedCallback() will be invoked. The callback
 * has the option to cause the session to fail with
 * ::VNCViewerErrorNotLicensedForServer.
 *
 * In a fully closed system, both the viewer and server applications define a
 * pair of features, X and Y, with the viewer providing feature X and the
 * server providing feature Y. The viewer then adds a remote feature check for
 * feature Y and the server adds a remote feature check for feature X. In this
 * way, those viewers and servers can only establish sessions with each other.
 *
 * A feature definition consists of a feature ID and a feature key. A feature
 * ID is a 32-bit unsigned integer. Contact your RealVNC sales representative
 * to have a range of feature IDs reserved for you.
 *
 * A feature key consists of 16 bytes. You should generate these randomly and
 * then code them into your viewer and server applications.
 *
 * The Viewer SDK must be licensed in order to enable the use of certain
 * features and to connect to certain different types of VNC Servers. To obtain
 * a VNC license, contact your RealVNC sales representative.
 *
 * Calls to VNCViewerAddLicenseFeature() must be made before starting the
 * viewer thread.
 *
 * \param pViewer The VNC Viewer to which to add the licensed feature.
 * \param featureID The feature ID.
 * \param featureKey The feature key.
 * \param featureKeySize The size in bytes of the buffer pointed to by
 * featureKey. This must be exactly 16.
 *
 * \retval VNCViewerErrorNone The licensed feature was defined successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The feature cannot be defined
 * because the viewer thread is already running.
 * \retval VNCViewerErrorInvalidParameter featureKey is NULL, or featureKeySize
 * is not 16. This error is also returned if the application attempts to define
 * a feature whose ID lies in the range that is reserved for use by RealVNC.
 *
 * \see VNCViewerProvideLicenseFeature(), VNCViewerAddRemoteFeatureCheck()
 */
typedef VNCViewerError VNCCALL
VNCViewerAddLicenseFeature(VNCViewer *pViewer,
                           vnc_uint32_t featureID,
                           const vnc_uint8_t *featureKey,
                           size_t featureKeySize);

/**
 * \brief Makes a licensed feature available to answer remote feature checks
 * made by the VNC Server.
 *
 * Calls to VNCViewerProvideLicenseFeature() must be made before starting the
 * viewer thread.
 *
 * For an overview of remote feature checks, see VNCViewerAddLicenseFeature().
 *
 * \param pViewer The VNC Viewer that should provide the licensed feature to
 * VNC Servers.
 * \param featureID The ID of the feature that should be provided.
 *
 * \retval VNCViewerErrorNone The licensed feature was provided successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The feature cannot be provided
 * because the viewer thread is already running.
 * \retval VNCViewerErrorInvalidParameter The feature ID lies in the range that
 * is reserved for use by RealVNC.
 *
 * \see VNCViewerAddLicenseFeature(), VNCViewerAddRemoteFeatureCheck()
 */
typedef VNCViewerError VNCCALL
VNCViewerProvideLicenseFeature(VNCViewer *pViewer,
                               vnc_uint32_t featureID);

/**
 * \brief Schedules a remote feature check to be made against the VNC Server in
 * all future sessions.
 *
 * The featureIDs and featureIDCount parameters specify an array of feature
 * IDs. In all future sessions with VNC Mobile Servers, the server will be
 * challenged to provide evidence that it has the correct feature key for at
 * least one of these features.
 *
 * In most uses, featureIDCount will be 1. However, you may wish to allow the
 * server a choice of features to provide if:
 *
 *  - You wish to allow interoperability with servers from multiple vendors.
 *  - You wish to offer different levels of functionality based on the
 *    particular feature that the server provides (e.g. you define 'premium'
 *    and 'basic' features, and enable certain functionality only if the server
 *    responds with the 'premium' feature).
 *
 * During session startup, if the server responds correctly to the remote
 * feature check succeeds, then the
 * VNCViewerRemoteFeatureCheckSucceededCallback() is invoked. If the server is
 * unable to answer the challenge, or answers the challenge incorrectly, then
 * the VNCViewerRemoteFeatureCheckFailedCallback() is invoked. (There is no
 * guarantee that exactly one of these callbacks is invoked for each remote
 * feature check; for example, if the session is lost during the licensing
 * handshake, then it is possible that neither will be invoked.)
 *
 * For an overview of remote feature checks, see VNCViewerAddLicenseFeature().
 *
 * \param pViewer The VNC Viewer for which the remote feature check should be
 * scheduled.
 * \param featureIDs The array of feature IDs.
 * \param featureIDCount The number of feature IDs in the array.
 * \param pFeatureCheckID On successful return, *pFeatureCheckID is an
 * SDK-assigned identifier for the feature check. This feature check ID is
 * passed to the VNCViewerRemoteFeatureCheckSucceededCallback() and
 * VNCViewerRemoteFeatureCheckFailedCallback().
 *
 * \retval VNCViewerErrorNone The licensed feature was provided successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The remote feature check cannot be
 * scheduled because the viewer thread is already running.
 * \retval VNCViewerErrorInvalidParameter featureIDS is NULL, or featureIDCount
 * is zero, or pFeatureCheckID is NULL.
 *
 * \see VNCViewerAddLicenseFeature(), VNCViewerProvideLicenseFeature(),
 * VNCViewerRemoteFeatureCheckSucceededCallback(),
 * VNCViewerRemoteFeatureCheckFailedCallback()
 */
typedef VNCViewerError VNCCALL
VNCViewerAddRemoteFeatureCheck(VNCViewer *pViewer,
                               const vnc_uint32_t *featureIDs,
                               size_t featureIDCount,
                               vnc_uint32_t *pFeatureCheckID);

/**
 * \brief Registers a bearer that is statically linked into the application.
 *
 * The preferred way to implement a bearer is to link it into a DLL or shared
 * object that exports the VNCBearerInitialize() entry point. This allows reuse
 * of the bearer from any Viewer SDK application, without recompilation.
 *
 * VNCViewerRegisterBearer() is provided for use when that approach is not
 * possible (e.g. if the platform on which the Viewer SDK is running does not
 * support runtime dynamic linking).
 *
 * The following static initialization functions are provided for the sample
 * bearers: VNCBearerInitialize_C, VNCBearerInitialize_D and
 * VNCBearerInitialize_L. VNC_BEARER_STATIC must be defined to enable
 * compilation of these functions.
 *
 * Legal bearer names are strings of between 1 and 16 characters in length,
 * consisting only of the capital letters A-Z, the decimal digits 0-9, and
 * underscores. The names of bearers authored by parties other than RealVNC
 * should begin with an underscore.
 *
 * \param pViewer The VNC Viewer for which the bearer should be registered.
 * \param bearerName The name of the bearer.
 * \param pBearerInitialize The address of the bearer initialization function.
 *
 * \retval VNCViewerErrorNone The bearer was registered successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The bearer cannot be registered
 * because the viewer thread is running.
 * \retval VNCViewerErrorAlreadyExists A bearer with this name has already been
 * registered.
 *
 * \see vncbearer.h, VNCBearerInitialize()
 */
typedef VNCViewerError VNCCALL
VNCViewerRegisterBearer(VNCViewer *pViewer,
                        const char *bearerName,
                        VNCBearerInitializeType *pBearerInitialize);

/**
 * \brief Requests that the server scales its framebuffer before sending it to
 * the viewer.
 *
 * This method causes all future sessions to request that the VNC Server scales
 * its framebuffer to the dimensions given by the width and height parameters.
 *
 * \note Currently <b>server-side scaling is supported only
 * by VNC Mobile Server for Android</b>. With Android servers, server-side
 * downscaling is known to provide significant performance gains across a
 * variety of handsets.
 *
 * \note However, while it is likely that equivalent
 * functionality will be added to VNC Mobile Server for other platforms in
 * future releases, <b>it cannot be guaranteed at this time that use of
 * server-side scaling on other platforms will not degrade
 * performance</b>.
 * Therefore, if it is not easy for you to update your viewer software once it
 * has been deployed, it is recommended that current viewer applications use
 * their VNCViewerServerInitCallback to query ::VNCViewerPropertyServerHostOS,
 * and call VNCViewerSetServerSideScaling() only if both the following are
 * true:
 *
 *  \note
 *  -# the server's display size, when rotated to match the desired
 *     orientation, will be larger than the viewer's display size
 *  -# the value of the ::VNCViewerPropertyServerHostOS property is not NULL
 *     and contains the substring 'android' (case insensitive)
 *
 * VNCViewerSetServerSideScaling() works with MirrorLink servers by adjusting
 * the way in which the ClientDisplayConfiguration structure is initialized
 * before the VNCViewerServerDisplayConfigurationCallback() is invoked.
 * However, in the MirrorLink case you may prefer to have your viewer
 * application perform this step itself in the
 * VNCViewerServerDisplayConfigurationCallback(), instead of using
 * VNCViewerSetServerSideScaling().
 *
 * Calling VNCViewerSetServerSideScaling() with the preserveAspectRatio flag
 * non-zero instructs the VNC Server to scale its display so that it is as
 * large as possible with neither the width nor the height of the scaled
 * display exceeding the requested width and height, but with the aspect ratio
 * preserved.
 *
 * Calling VNCViewerSetServerSideScaling() with the preserveAspectRatio flag
 * equal to zero instructs the VNC Server to scale its display so that its size
 * is exactly that requested. The width and height may be scaled by different
 * factors to achieve this.
 *
 * VNCViewerSetServerSideScaling() may be called before the viewer thread is
 * started, or during a session. A good time to call it is in the
 * VNCViewerServerInitCallback, when the size of the server's display first
 * becomes known.
 *
 * A successful call to this method is therefore <b>not</b> a guarantee that
 * the server will be able to scale to the requested dimensions.  The size of
 * the required framebuffer is always given by the VNCViewerServerInitCallback
 * or the last received VNCViewerDesktopResizeCallback. If the server is not
 * able to satisfy a server-side scaling request, then the width and height in
 * these callbacks may not match those passed to
 * VNCViewerSetServerSideScaling().  In this case, it is at the viewer
 * application's discretion how it should proceed. (For example, the
 * application may fall back to scaling at the viewer-side, or it may clip the
 * framebuffer so that it fits in the available display area).
 *
 * Calling VNCViewerSetServerSideScaling() with width and height both equal to
 * zero causes future sessions to be unscaled.
 *
 * \param pViewer The VNC Viewer which is to request server-side scaling.
 * \param width The width to which the server should scale its display.
 * \param height The height to which the server should scale its display.
 * \param preserveAspectRatio Whether the server should preserve its aspect
 * ratio.
 *
 * \retval VNCViewerErrorNone The operation was successful.
 * running.
 * \retval VNCViewerErrorInvalidParameter One or more of the parameters is
 * invalid.
 */
typedef VNCViewerError VNCCALL
VNCViewerSetServerSideScaling(VNCViewer *pViewer,
                              vnc_uint32_t width,
                              vnc_uint32_t height,
                              vnc_uint8_t preserveAspectRatio);

/**
 * \brief Registers a pluggable decoder with the SDK.
 *
 * The decoder is specified by pDecoderInitialize, a pointer to a function
 * which generates a factory, allowing the SDK to generate one or more
 * decoders of that type.
 *
 * Only decoders supporting RAW, TRLE, ZRLE, JPEG, SLRLE, H264 and VAH264
 * will be allowed.
 * Attempts to register or load a decoder which supports a different encoding
 * will fail. Additionally, attempting to register more than one decoder for
 * a particular encoding will result in the second encoder failing to
 * register.
 *
 * The VNCDecoderViewerContext passed to this method will be passed unaltered
 * to VNCDecoderFactoryInitialize() call. This allows viewers and decoders to
 * share a common context object, containing anything that both need to ensure
 * the correct display of any items on the screen.
 *
 * It should be noted that the pluggable decoders API is not thread safe. A
 * decoder wishing to make use of the supporting API methods must do so from
 * the viewer SDK thread.
 *
 * \param pViewer The VNC Viewer which is requesting the registration of the
 * decoder.
 * \param pDecoderInitialize A pointer to the factory initialization function.
 * \param pDecoderViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the VNCDecoderFactoryInitialize
 * function.
 *
 * \return VNCViewerErrorNone if the registration was successful, or an
 * appropriate error message if it was not successful.
 */
typedef VNCViewerError VNCCALL
VNCViewerRegisterDecoder(VNCViewer* pViewer,
                     VNCDecoderFactoryInitializeType *pDecoderInitialize,
                     VNCDecoderViewerContext pDecoderViewerContext);

/**
 * \brief Registers a pluggable decoder with the SDK.
 *
 * The decoder is specified by the decoderName parameter -- the SDK will
 * look in the directory specified in the VNCParameterDecoderDirectory
 * SDK parameter, and try to load a decoder from a shared library with the
 * following filename (where 'NAME' is the decoderName argument):
 *
 *   On Linux:    'libvncdecoder-NAME.so'
 *   On Mac OS X: 'libvncdecoder-NAME.dylib'
 *   On Windows:  'vncdecoder-NAME.dll'
 *
 * Only decoders supporting RAW, TRLE, ZRLE, JPEG, SLRLE, H264 and VAH264 will
 * be allowed.
 * Attempts to register or load a decoder which supports a different encoding
 * will fail. Additionally, attempting to register more than one decoder for
 * a particular encoding will result in the second encoder failing to
 * register.
 *
 * The VNCDecoderViewerContext passed to this method will be passed unaltered
 * to VNCDecoderFactoryInitialize() call. This allows viewers and decoders to
 * share a common context object, containing anything that both need to ensure
 * the correct display of any items on the screen.
 *
 * It should be noted that the pluggable decoders API is not thread safe. A
 * decoder wishing to make use of the supporting API methods must do so from
 * the viewer SDK thread.
 *
 * \param pViewer The VNC Viewer which is requesting the registration of the
 * decoder.
 * \param pDecoderInitialize A pointer to the factory initialization function.
 * \param pDecoderViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the VNCDecoderFactoryInitialize
 * function.
 *
 * \return VNCViewerErrorNone if the registration was successful, or an
 * appropriate error message if it was not successful.
 */
typedef VNCViewerError VNCCALL
VNCViewerLoadDecoderFromFile(VNCViewer* pViewer,
                     const char *decoderName,
                     VNCDecoderViewerContext pDecoderViewerContext);

/**
 * \brief Registers a pluggable renderer with the SDK.
 *
 * The renderer is specified by pRendererInitialize, a pointer to a function
 * which generates a factory, allowing the SDK to generate a renderer of that
 * type when a connection is started.
 *
 * Only one pluggable renderer may be loaded at any time.
 *
 * The VNCRendererViewerContext passed to this method will be passed unaltered
 * to VNCRendererFactoryInitialize() call. This allows viewers and renderers to
 * share a common context object, containing anything that both need to ensure
 * the correct display of any items on the screen.
 *
 * It should be noted that the pluggable renderers API is not thread safe. A
 * renderer wishing to make use of the supporting API methods must do so from
 * the viewer SDK thread.
 *
 * This method should not be called while a connection is active.
 *
 * This method will fail if a skin is loaded.
 *
 * \param pViewer The VNC Viewer which is requesting the registration of the
 * renderer.
 * \param pRendererInitialize A pointer to the factory initialization function.
 * \param pRendererViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the VNCRendererFactoryInitialize
 * function.
 *
 * \return VNCViewerErrorNone if the registration was successful, or an
 * appropriate error message if it was not successful.
 */
typedef VNCViewerError VNCCALL
VNCViewerRegisterRenderer(VNCViewer* pViewer,
                          VNCRendererFactoryInitializeType *pRendererInitialize,
                          VNCRendererViewerContext pRendererViewerContext);

/**
 * \brief Registers a pluggable renderer with the SDK.
 *
 * The renderer is specified by the rendererName parameter -- the SDK will
 * look in the directory specified in the VNCParameterRendererDirectory
 * SDK parameter, and try to load a renderer from a shared library with the
 * following filename (where 'NAME' is the rendererName argument):
 *
 *   On Linux:    'libvncrenderer-NAME.so'
 *   On Mac OS X: 'libvncrenderer-NAME.dylib'
 *   On Windows:  'vncrenderer-NAME.dll'
 *
 * The VNCRendererViewerContext passed to this method will be passed unaltered
 * to VNCRendererFactoryInitialize() call. This allows viewers and renderers to
 * share a common context object, containing anything that both need to ensure
 * the correct display of any items on the screen.
 *
 * It should be noted that the pluggable renderers API is not thread safe. A
 * renderer wishing to make use of the supporting API methods must do so from
 * the viewer SDK thread.
 *
 * This method should not be called while a connection is active.
 *
 * This method will fail if a skin is loaded.
 *
 * \param pViewer The VNC Viewer which is requesting the registration of the
 * renderer.
 * \param pRendererInitialize A pointer to the factory initialization function.
 * \param pRendererViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the VNCRendererFactoryInitialize
 * function.
 *
 * \return VNCViewerErrorNone if the registration was successful, or an
 * appropriate error message if it was not successful.
 */
typedef VNCViewerError VNCCALL
VNCViewerLoadRendererFromFile(VNCViewer* pViewer,
                              const char *rendererName,
                              VNCRendererViewerContext pRendererViewerContext);

/**
 * \brief Sets a property of a pluggable renderer.
 *
 * For example, viewer applications may use this method to set the
 * visibility, position, and size of a pluggable renderer's overlay.
 *
 * The arguments to this method will be passed directly to the renderer's
 * VNCRendererPropertySet method.
 *
 * If there is no pluggable renderer loaded, or the pluggable renderer failed
 * to initialise, calls to this method will be ignored. This method has no
 * side effects other than passing the key and value directly to the loaded
 * pluggable renderer.
 *
 * \param pViewer The VNC Viewer which is requesting the change.
 * \param propertyKey The key representing the property being set.
 * \param newValue A pointer to the new value of the property. This is owned
 * by the viewer application, and should be valid for the duration of
 * this method call.
 * \param valueSize The size of the value pointed to by newValue, in bytes.
 * \return VNCViewerErrorNone if the change was successful, or an
 * appropriate error message if it was not successful.
 */
typedef VNCViewerError VNCCALL
VNCViewerSetRendererProperty(VNCViewer* pViewer,
                             VNCRendererProperty propertyKey,
                             const void* newValue,
                             size_t valueSize);

/**
 * \brief Registers a bearer with a dynamic context.
 *
 * The preferred way to implement a bearer is to link it into a DLL or shared
 * object that exports the VNCBearerInitialize() entry point. This allows reuse
 * of the bearer from any Viewer SDK application, without recompilation.
 *
 * VNCViewerRegisterBearerEx() is provided for use when that approach is not
 * possible (e.g. if the platform on which the Viewer SDK is running does not
 * support runtime dynamic linking).
 *
 * The following static initialization functions are provided for the sample
 * bearers: VNCBearerInitialize_C, VNCBearerInitialize_D and
 * VNCBearerInitialize_L. VNC_BEARER_STATIC must be defined to enable
 * compilation of these functions.
 *
 * Legal bearer names are strings of between 1 and 16 characters in length,
 * consisting only of the capital letters A-Z, the decimal digits 0-9, and
 * underscores. The names of bearers authored by parties other than RealVNC
 * should begin with an underscore.
 * 
 * This method has an additional parameter for the bearer's dynamic context,
 * which is a pointer passed through the SDK to the bearer (the bearer
 * accesses this by calling a support API). Hence this API provides a means
 * to share runtime state with the initialised bearer.
 *
 * \param pViewer The VNC Viewer for which the bearer should be registered.
 * \param bearerName The name of the bearer.
 * \param pBearerInitialize The address of the bearer initialization function.
 * \param pBearerDynamicContext A pointer to the bearer's dynamic context.
 *
 * \retval VNCViewerErrorNone The bearer was registered successfully.
 * \retval VNCViewerErrorIllegalWhileRunning The bearer cannot be registered
 * because the viewer thread is running.
 * \retval VNCViewerErrorAlreadyExists A bearer with this name has already been
 * registered.
 *
 * \see vncbearer.h, VNCBearerInitialize(), VNCBearerGetDynamicContext()
 */
typedef VNCViewerError VNCCALL
VNCViewerRegisterBearerEx(VNCViewer *pViewer,
                          const char *bearerName,
                          VNCBearerInitializeType *pBearerInitialize,
                          VNCBearerDynamicContext pBearerDynamicContext);

/**
 * \brief Returns a symbolic name corresponding to a Viewer SDK error code.
 *
 * For example, if error is VNCViewerErrorInvalidParameter, then the returned
 * string is "InvalidParameter". Such strings may be useful as
 * internationalization keys or in log entries.
 *
 * The strings returned by VNCGetErrorName() are brief and are not
 * internationalized. They are therefore not suitable for display to an end
 * user.
 *
 * \param error The error code.
 * \return A short string corresponding to the error code. The memory pointed
 * to is owned by the SDK; do not free it. If the error code is not recognized
 * by the SDK, then the return value is NULL.
 */
typedef const char *VNCCALL
VNCGetErrorName(VNCViewerError error);

/**
 * \brief Uninitializes the SDK.
 *
 * You must call this function after destroying all VNC Viewers but before
 * unloading the SDK.
 *
 * \see VNCViewerSDKInitialize, VNCViewerDestroy
 */
typedef void VNCCALL
VNCViewerSDKUninitialize(void);

/**
 * \brief Contains the SDK's version and the addresses of the SDK's methods.
 *
 * An instance of this structure should be passed to VNCViewerSDKInitialize().
 * A successful call to VNCViewerSDKInitialize() fills in the structure with
 * the SDK's major and minor version numbers and the addresses of the SDK's VNC
 * viewer methods.
 *
 * After the structure is initialized, you may create a VNC Viewer and interact
 * with it by calling the SDK's methods through the function pointers in the
 * structure.  Begin by calling VNCViewerCreate().
 *
 * You may also wish to check the version numbers reported by the SDK.  These
 * numbers are the version of the SDK itself, not the version of the RFB
 * protocol.
 *
 * For full documentation on each of the function pointers in the VNCViewerSDK
 * structure, see the documentation for the corresponding typedef.
 */
typedef struct
{
    /** Major component of the SDK's version number. */
    vnc_int32_t versionMajor;
    /** Minor component of the SDK's version number. */
    vnc_int32_t versionMinor;
    /** Patch component of the SDK's version number. */
    vnc_int32_t versionPatch;
    /** Build number component of the SDK's version number. */
    vnc_int32_t versionBuild;

    /** Uninitializes the SDK. */
    VNCViewerSDKUninitialize *vncViewerSDKUninitialize;
    /** Allocates a string for use by the SDK. */
    VNCAllocString *vncAllocString;
    /** Frees a string that was allocated with VNCAllocString. */
    VNCFreeString *vncFreeString;
    /** Creates a VNC Viewer. */
    VNCViewerCreate *vncViewerCreate;
    /** Destroys a VNC Viewer. */
    VNCViewerDestroy *vncViewerDestroy;
    /** Attempts to connect to a VNC server. */
    VNCViewerConnect *vncViewerConnect;
    /** Stops a VNC Viewer. */
    VNCViewerReset *vncViewerReset;
    /** Queries a VNC Viewer for the value of a runtime property. */
    VNCViewerGetProperty *vncViewerGetProperty;
    /** Queries a VNC Viewer for the value of parameter. */
    VNCViewerGetParameter *vncViewerGetParameter;
    /** Sets the value of a VNC Viewer parameter. */
    VNCViewerSetParameter *vncViewerSetParameter;
    /** Iterates the names of the supported VNC Viewer parameters. */
    VNCViewerIterateParameterNames *vncViewerIterateParameterNames;
    /** Assigns a context data pointer to a VNC Viewer. */
    VNCViewerSetContext *vncViewerSetContext;
    /** Retrieves a context data pointer from a VNC Viewer. */
    VNCViewerGetContext *vncViewerGetContext;
    /** Sets the session pixel format. */
    VNCViewerSetPixelFormat *vncViewerSetPixelFormat;
    /** Translates a platform-specific special key symbol into an X key symbol. */
    VNCSpecialKeyToXKeySymbol *vncSpecialKeyToXKeySymbol;
    /** Translates an X key symbol into a platform-specific special key symbol. */
    VNCXKeySymbolToSpecialKey *vncXKeySymbolToSpecialKey;
    /** Translates a Unicode character into an X key symbol. */
    VNCUnicodeToXKeySymbol *vncUnicodeToXKeySymbol;
    /** Translates an X key symbol into a Unicode character. */
    VNCXKeySymbolToUnicode *vncXKeySymbolToUnicode;
    /** Causes a KeyEvent RFB message to be sent to the VNC server. */
    VNCViewerSendKeyEvent *vncViewerSendKeyEvent;
    /** Causes a PointerEvent RFB message to be sent to the VNC server. */
    VNCViewerSendPointerEvent *vncViewerSendPointerEvent;
    /** Generates an RSA public and private key pair for use by a VNC Viewer. */
    VNCGenerateRSAKeys *vncGenerateRSAKeys;
    /** Enables or disables display updates. */
    VNCViewerEnableDisplayUpdates *vncViewerEnableDisplayUpdates;
    /** Listens for incoming connections from a VNC server. */
    VNCViewerListen *vncViewerListen;
    /** Parses the given VNC Command String and then executes the command. */
    VNCViewerProcessCommandString *vncViewerProcessCommandString;
    /**
     * Causes a DeviceKey RFB DeviceExtension message to be sent to the VNC
     * server.
     */
    VNCViewerSendDeviceKeyEvent *vncViewerSendDeviceKeyEvent;
    /**
     * Processes Windows key event messages and coverts them to RFB KeyEvent
     * messages.
     */
    VNCViewerProcessWindowsKeyMessage *vncViewerProcessWindowsKeyMessage;
    /**
     * 'Undoes' KeyEvent and DeviceKey messages that have been generated by
     * VNCViewerProcessWindowsKeyMessage().
     */
    VNCViewerReleaseAllKeys *vncViewerReleaseAllKeys;
    /**
     * Configures the encryption and authentication mechanisms used by the VNC
     * viewer.
     */
    VNCViewerSetSecurityTypes *vncViewerSetSecurityTypes;
    /**
     * Causes a device key or KeyEvent RFB message to be sent to the VNC
     * server.
     */
    VNCViewerSendDeviceKeyOrKeyEvent *vncViewerSendDeviceKeyOrKeyEvent;
    /** Selects a mobile device skin for use with a VNC session. */
    VNCViewerSetSkin *vncViewerSetSkin;
    /** Registers a VNCViewerSkinFetchCallback() with a VNC Viewer object. */
    VNCViewerAddSkinFetchCallback *vncViewerAddSkinFetchCallback;
    /** Unregisters a VNCViewerSkinFetchCallback() from a VNC Viewer object. */
    VNCViewerRemoveSkinFetchCallback *vncViewerRemoveSkinFetchCallback;
    /**
     * Unregisters all VNCViewerSkinFetchCallback() from a VNC Viewer object.
     */
    VNCViewerRemoveAllSkinFetchCallbacks *vncViewerRemoveAllSkinFetchCallbacks;
    /**
     * Notifies the Viewer SDK that the pending VNCViewerSkinFetchCallback()
     * has completed.
     */
    VNCViewerSkinFetchCallbackComplete *vncViewerSkinFetchCallbackComplete;
    /**
     * Queries a VNC Viewer for the value of a string-valued runtime property.
     */
    VNCViewerGetPropertyString *vncViewerGetPropertyString;
    /** Causes a ClientCutText RFB message to be sent to the VNC server. */
    VNCViewerSendClientCutText *vncViewerSendClientCutText;
    /**
     * Causes a FramebufferUpdateRequest RFB message to the sent to the VNC
     * server.
     */
    VNCViewerFrameBufferUpdateRequest *vncViewerFrameBufferUpdateRequest;
    /**
     * Withdraws a FramebufferUpdateRequest that was previously sent to the VNC
     * server.
     */
    VNCViewerFrameBufferUpdateCancel *vncViewerFrameBufferUpdateCancel;
    /**
     * Notifies the Viewer SDK of the result of a
     * VNCViewerAsynchronousCredentialsCallback.
     */
    VNCViewerAsynchronousCredentialsResult *vncViewerAsynchronousCredentialsResult;
    /**
     * Notifies the Viewer SDK of the result of a
     * VNCViewerAsynchronousValidatePasswordCallback.
     */
    VNCViewerAsynchronousValidatePasswordResult *vncViewerAsynchronousValidatePasswordResult;
    /**
     * Registers an application-defined protocol extension with the Viewer SDK.
     */
    VNCViewerRegisterExtension *vncViewerRegisterExtension;
    /** Sends a message on an application-defined protocol extension. */
    VNCViewerSendExtensionMessage *vncViewerSendExtensionMessage;
    /**
     * Sends a DeviceStatusRequest MirrorLink extension message to the server.
     */
    VNCViewerSendDeviceStatusRequest
        *vncViewerSendDeviceStatusRequest;
    /**
     * Sends a FramebufferBlockingNotification MirrorLink extension message to
     * the server.
     */
    VNCViewerSendFramebufferBlockingNotification
        *vncViewerSendFramebufferBlockingNotification;
    /**
     * Sends an AudioBlockingNotification MirrorLink extension message to the
     * server.
     */
    VNCViewerSendAudioBlockingNotification
        *vncViewerSendAudioBlockingNotification;

    /**
     * \brief Notifies the Viewer SDK of the result of a
     * VNCViewerAsynchronousVerifyServerIdentityCallback.
     */
    VNCViewerAsynchronousVerifyServerIdentityResult
        *vncViewerAsynchronousVerifyServerIdentityResult;

    /**
     * \brief Queries the session pixel format.
     */
    VNCViewerGetPixelFormat *vncViewerGetPixelFormat;

    /**
     * \brief Notifies the viewer SDK that the viewer application has rendered
     * the framebuffer.
     */
    VNCViewerFrameBufferRendered *vncViewerFrameBufferRendered;

    /** Translates from an encoding identifier to a string containing its name. */
    VNCGetEncodingName *vncGetEncodingName;

    /** Adds a VNC license to the VNC Viewer. */
    VNCViewerAddLicense *vncViewerAddLicense;

    /** Defines a licensed feature on the VNC Viewer. */
    VNCViewerAddLicenseFeature *vncViewerAddLicenseFeature;

    /**
     * \brief Makes a licensed feature available to answer remote feature
     * checks made by the VNC Server.
     */
    VNCViewerProvideLicenseFeature *vncViewerProvideLicenseFeature;

    /**
     * \brief Schedules a remote feature check to be made against the VNC
     * Server in all future sessions.
     */
    VNCViewerAddRemoteFeatureCheck *vncViewerAddRemoteFeatureCheck;

    /** Register an external VNCBearerInitialize function. */
    VNCViewerRegisterBearer *vncViewerRegisterBearer;

    /**
     * \brief Associates a context pointer with an application-defined protocol
     * extension.
     */
    VNCViewerSetExtensionContext *vncViewerSetExtensionContext;

    /**
     * \brief Retrieve the context pointer with an application-defined protocol
     * extension.
     */
    VNCViewerGetExtensionContext *vncViewerGetExtensionContext;

    /**
     * \brief Requests that the server scales its framebuffer before sending it
     * to the viewer.
     */
    VNCViewerSetServerSideScaling *vncViewerSetServerSideScaling;

    /** Sets a configuration for a bearer used by the VNC Viewer. */
    VNCViewerSetBearerConfiguration *vncViewerSetBearerConfiguration;

    /** Performs a local feature check. */
    VNCViewerLocalFeatureCheck *vncViewerLocalFeatureCheck;

    /** Sets the framebuffer alpha mask */
    VNCViewerSetFrameBufferAlphaMask *vncViewerSetFrameBufferAlphaMask;

    /** Returns a symbolic name corresponding to a Viewer SDK error code. */
    VNCGetErrorName *vncGetErrorName;

    /** Sends a multi-touch event to the VNC Server. */
    VNCViewerSendTouchEvent *vncViewerSendTouchEvent;

    /** Informs the SDK that the VNC Viewer has lost the input focus. */
    VNCViewerLostFocus *vncViewerLostFocus;

    /**
     * Translates a Unicode character into an X key symbol, taking account of
     * the type of the VNC Server to which the given VNC Viewer is connected.
     */
    VNCViewerUnicodeToXKeySymbol *vncViewerUnicodeToXKeySymbol;

    /** Sends a VirtualKeyboardTriggetRequest MirrorLink extension
     * message to the server. */
    VNCViewerSendVirtualKeyboardTriggerRequest *vncViewerSendVirtualKeyboardTriggerRequest;

    /** Sends an EventMappingRequest MirrorLink extension
     * message to the server. */
    VNCViewerSendEventMappingRequest *vncViewerSendEventMappingRequest;

    /** Sends a KeyEventListingRequest MirrorLink extension message to
     * the server. */
    VNCViewerSendKeyEventListingRequest *vncViewerSendKeyEventListingRequest;

    /**
     * Informs the SDK of the outcome of a
     * VNCViewerContentAttestationFailureCallback.
     */
    VNCViewerContentAttestationFailureResult *vncViewerContentAttestationFailureResult;

    /**
     * Sends a FramebufferBlockingNotification MirrorLink extension message to
     * the server as soon as the client is allowed to send.
     */
    VNCViewerSendFramebufferBlockingNotificationEx *vncViewerSendFramebufferBlockingNotificationEx;

    /**
     * Cancels all the framebuffer blocking notifications that are queued to be
     * sent to the server.
     */
    VNCViewerCancelFramebufferBlockingNotifications *vncViewerCancelFramebufferBlockingNotifications;

    /**
     * \brief Queries the pixel format currently in use.
     */
    VNCViewerGetActivePixelFormat* vncViewerGetActivePixelFormat;

    /**
     * Attempts to register a pluggable decoder using the supplied function
     * pointer.
     */
    VNCViewerRegisterDecoder *vncViewerRegisterDecoder;

    /**
     * Attempts to register a pluggable decoder with the specified name, from
     * an appropriately named file.
     */
    VNCViewerLoadDecoderFromFile *vncViewerLoadDecoderFromFile;

    /**
     * Attempts to register a pluggable renderer using the supplied function
     * pointer.
     */
    VNCViewerRegisterRenderer *vncViewerRegisterRenderer;

    /**
     * Attempts to register a pluggable renderer with the specified name, from
     * an appropriately named file.
     */
    VNCViewerLoadRendererFromFile *vncViewerLoadRendererFromFile;

    /**
     * Sets a pluggable renderer property.
     */
    VNCViewerSetRendererProperty *vncViewerSetRendererProperty;

    /**
     * Registers a bearer with a dynamic context.
     */
    VNCViewerRegisterBearerEx *vncViewerRegisterBearerEx;
} VNCViewerSDK;

/**
 * \brief The type of VNCViewerSDKInitialize().
 *
 * If you are dynamically loading the SDK at runtime, you should use this
 * typedef to declare your function pointer.
 *
 * \see VNCViewerSDKInitialize
 */
typedef VNCViewerError VNCCALL
VNCViewerSDKInitializeType(VNCViewerSDK *pSDK, size_t sdkSize);

/**
 * \brief The entry point to the SDK.
 *
 * This function is the sole entry point exported by the SDK.  Before you can
 * call any other of the SDK's functions, you must call this function to obtain
 * pointers to their implementations.
 *
 * Before unloading the SDK, you should call VNCViewerSDKUninitialize().
 *
 * \param pSDK Pointer to the VNCViewerSDK structure to be initialized.
 * \param sdkSize Size of the VNCViewerSDK structure to be initialized.
 *
 * \return VNCViewerErrorNone if successful, or an explanatory error code
 * otherwise.
 *
 * \see VNCViewerSDK, VNCViewerSDKUninitialize, VNCViewerSDKInitializeType,
 * VNCVerifyIntegerTypes
 */
VNCDLLIMPORT VNCViewerError VNCCALL
VNCViewerSDKInitialize(VNCViewerSDK *pSDK, size_t sdkSize);

/**
 * \brief Allows the application to verify that its use of VNC_USE_STDINT_H or
 * VNC_USE_DEFAULT_INT_TYPEDEFS is correct.
 *
 * This is an alias for VNCVerifyIntegerTypes() that is provided to maintain
 * backwards compatibility.
 *
 * \see VNCVerifyIntegerTypes
 */
#define VNCViewerSDKVerifyIntegerTypes() VNCVerifyIntegerTypes()

/**
 * \brief Wrapper macro for defining a color map based pixel format.
 *
 * \see VNCPixelFormat
 */
#define VNCPixelFormatIndexed(bitDepth, bigEndianFlag) \
  {(bitDepth > 16 ? 32 : bitDepth > 8 ? 16 : 8), \
   bitDepth, bigEndianFlag, 0, 0, 0, 0, 0, 0, 0}

/**
 * \brief Wrapper macro for defining a RGB based pixel format.
 *
 * \see VNCPixelFormat
 */
#define VNCPixelFormatRGB(redBitDepth, greenBitDepth, blueBitDepth, \
                          bigEndianFlag) \
  {((redBitDepth + greenBitDepth + blueBitDepth) > 16) ? 32 : \
   ((redBitDepth + greenBitDepth + blueBitDepth) > 8) ? 16 : 8, \
   (redBitDepth + greenBitDepth + blueBitDepth), \
   bigEndianFlag, \
   1, /*true colour*/ \
   (!(redBitDepth > 0 )? 0: (2 << (redBitDepth - 1)) - 1), \
   (!(greenBitDepth > 0)? 0: (2 << (greenBitDepth - 1)) - 1), \
   (!(blueBitDepth > 0)? 0: (2 << (blueBitDepth - 1)) - 1), \
   (greenBitDepth + blueBitDepth), \
   blueBitDepth, \
   0}

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCVIEWERSDK_H__) */
